
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DaveopDataConcatenate
;
; PURPOSE:
;   A simple operation for concatenating two or more datasets together. 
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   ResetOpAttributes
;   SetProperty
;   UndoOperation
;
; Philip Tregenna-Piggott
; Paul Scherrer Institut
; Switzerland.
; philip.tregenna@psi.ch
;
; Adapted from software written by
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; May 2009
;-
;###############################################################################


;===============================================================================
; DaveopDataConcatenate::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset1 [out] - Dataset A in 'A = A - B'
;
;   dataset2 [out] - Dataset B in 'A = A - B'
;
; RETURN VALUE:
;
pro DaveopDataConcatenate::GetProperty ,ndata=ndata,resultname=resultname,algorithm=algorithm $
                  ,_REF_EXTRA=etc
compile_opt idl2

;ndata
if (arg_present(ndata)) then ndata=self.ndata
;name of dataset to store result
if (arg_present(resultName)) then resultName=self.resultName
; Algorithm
if (arg_present(algorithm)) then algorithm=self.algorithm

; dataset* properties
nEtc = n_elements(etc)
if (nEtc gt 0) then begin
   propNames = Self.propertyNames
   for i=0, nEtc-1 do begin
      index = where(etc[i] eq strupcase(propNames), found)
      if (found) then (Scope_Varfetch(etc[i],/REF_EXTRA)) = Self.datasetsIndex[index]
   endfor
endif

; also call base class accessor
if(nEtc gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DaveopDataConcatenate::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset1 [out] - Dataset A in 'A = A - B'
;
;   dataset2 [out] - Dataset B in 'A = A - B'
;
; RETURN VALUE:
;
pro DaveopDataConcatenate::SetProperty,ndata=ndata,nmax=nmax,resultname=resultname,algorithm=algorithm $
                      ,_EXTRA=etc
compile_opt idl2

; Nmax
if (n_elements(nmax)) then self.nmax = nmax > 2 ; must be >=2

; Ndata
if (n_elements(ndata)) then begin
    self.ndata= ndata
    for i=0,self.nmax-1 do begin
        hide = (i ge self.ndata)
        propName = Self.propertyNames[i]
        self->SetPropertyAttribute, propName, hide=hide
    endfor
endif

; Name of dataset to store results
if (n_elements(resultName)) then begin
    self.resultName=resultName
    self.namechanged = 1
endif

; Algorithm
if (n_elements(algorithm)) then self.algorithm=algorithm

; dataset* properties
nEtc = n_elements(etc)
if (nEtc gt 0) then begin
   propNames = Self.propertyNames
   etcTags = tag_names(etc)
   for i=0, nEtc-1 do begin
      index = where(etcTags[i] eq strupcase(propNames), found)
      if (found) then  Self.datasetsIndex[index] = etc.(i) ;Scope_Varfetch(etc.(i))
   endfor
endif

; Also Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DaveopDataConcatenate::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case). Only included here to maintain the API
;                  of this method
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataConcatenate::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store the values
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, resultName=resultName, algorithm=algorithm
void = oCmd->AddItem('OLD_RESULTNAME',resultName)
void = oCmd->AddItem('OLD_ALGORITHM',algorithm)

; Add the command to command set
oCmdSet->Add, oCmd


return, 1

end


;===============================================================================
; DaveopDataConcatenate::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case). Only included here to maintain the API
;                  of this method
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataConcatenate::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, resultName=resultName,algorithm=algorithm
void = oCmd->AddItem('NEW_RESULTNAME',resultName)
void = oCmd->AddItem('NEW_ALGORITHM',algorithm)

return, 1

end


;===============================================================================
; DaveopDataConcatenate::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataConcatenate::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's properties.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DaveopDataConcatenate::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataConcatenate::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 2) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Restore operation attributes
void = oCmds[0]->GetItem('OLD_RESULTNAME',resultName)
void = oCmds[0]->GetItem('OLD_ALGORITHM',algorithm)
self->SetProperty, resultName=resultName, algorithm=algorithm

; Undo previous concatenate by deleting the previously generated output from the Data Manager
void = oCmds[1]->GetItem('FULLOUTPUTID',idDataset)
oDataset = oTool->GetByIdentifier(idDataset)
oDM = oTool->GetService("DAVEDATA_MANAGER")	
status = oDM->DeleteData(oDataset->GetFullIdentifier())

; Indicate that the tool has been modified and force it to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DaveopDataConcatenate::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataConcatenate::RedoOperation, oCmdSet
compile_opt idl2


; Retrieve the command objects
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 2) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmds[0]->GetProperty, target_identifier=idCmdTarget
oCmdTarget = oTool->GetByIdentifier(idCmdTarget)

; Restore operation attributes
void = oCmds[0]->GetItem('NEW_RESULTNAME',resultName)
void = oCmds[0]->GetItem('NEW_ALGORITHM',algorithm)
self->SetProperty, algorithm=algorithm, resultName=resultName

; Restrieve useful attributes 
void = oCmds[1]->GetItem('ALGORITHMNAME',algoName)
void = oCmds[1]->GetItem('IDDATASETS',idDatasets)
void = oCmds[1]->GetItem('DATASETNAMES',names)
void = oCmds[1]->GetItem('OUTPUTNAME',resName)
void = oCmds[1]->GetItem('OUTPUTID',resID)
nData = n_elements(idDatasets)

newTrmt = '______________________________'
newTrmt = [newTrmt,'Concatenate operation performed using "'+algoName+'" algorithm']
pointer=0
for i=0,nData-1 do begin
   idDataset = idDatasets[i]
   oDataset = oTool->GetByIdentifier(idDataset)
;print,'obj_valid(oDataset) ',obj_valid(oDataset)
   if (~obj_valid(oDataset)) then continue

   oDataset->GetProperty, dataRef=oDat, errorRef=oErr, axis1Ref=oInd  ;get the dependent and error in dependent data

   if (~obj_valid(oDat) || ~obj_valid(oErr)) then continue

   if (~oDat->GetData(dat) || ~oErr->GetData(err) || ~oInd->GetData(vin) ) then begin
       msg = 'Concatenate: Error retrieving data from dataset!'
       oTool->StatusMessage,msg
       return, 0
   endif

   if (i eq 0) then begin
       dS1 = size(dat)
       dS2 = size(err)
       dS3 = size(vin)
       ;; dims and nos of elements should be the same for data and error
       passed = ((dS1[0] eq dS2[0]) && (dS1[1] eq dS2[1]) && (dS1[2] eq dS2[2])) and $
                ((dS3[1] eq dS3[1]))
       if (~passed) then begin
           msg = 'Concatenate: Inconsistent dimensions and/or nos of elements!'
           oTool->StatusMessage,msg
           return, 0
       endif
       firstDataset = oDataset
   endif
   s1 = size(dat)
   s2 = size(err)
   s3 = size(vin)
   ;; dims and nos of elements should be the same for data and error
   passed = ((s1[0] eq s2[0]) && (s1[1] eq s2[1]) && (s1[2] eq s2[2])) and  $
            ( (s1[1] eq s3[1]) && s1[0] eq dS1[0] && s2[0] eq dS2[0] && s3[0] eq dS3[0]  ) 
    if (~passed) then begin
       msg = 'Concatenate: Inconsistent dimensions and/or nos of elements!'
       oTool->StatusMessage,msg
       return, 0
   endif

if i eq 0 then begin
data=dat & error=err & vout=vin & pointer=[pointer,dS3[1]]
endif else begin
data=[data,dat]
error=[error,err]
vout=[vout,vin]
pointer=[pointer,pointer[i]+S3[1]]
endelse
   newTrmt = [newTrmt,'Include "'+names[i]+'" in concatenation']
endfor

case strupcase(algoName) of
   'AUTO': begin         ;simple Concatenation
 order=sort(vout)
 data=data[order,*]
 error=error[order,*]
 vout=vout[order]
     end
   
   'USER INPUT': begin

;user previous selection  
data=(*self.Concatenate_StructurePtr).data
error=(*self.Concatenate_StructurePtr).error
vout=(*self.Concatenate_StructurePtr).vout
   end

   else: begin
       oTool->StatusMessage, 'Concatenation could not be performed - unknown algorithm!'
       return, 0
   end
endcase

if (obj_hasmethod(firstDataset,'Clone')) then begin
   ; use clone method to replicate first dataset
   oRes = firstDataset->Clone()
   oRes->SetProperty, name=resName, identifier=resID
   oTool->AddDatasetToDataManager, oRes
endif else begin
   ; use DM to replicate first dataset
   oDM = oTool->GetService("DAVEDATA_MANAGER")  ;; get DM service
   if (~oDM->CopyData(firstDataset->GetFullIdentifier(), name=resName, copyid=resID)) then return, 0
   oRes = oTool->GetByIdentifier(resID)
endelse

; Get the treatment info and update it
trmtExist = 0

oRes->GetProperty, trmtRef=oTrmt, dataRef=oDat, errorRef=oErr, axis1Ref=oInd
if (obj_valid(oTrmt)) then $
  trmtExist = oTrmt->GetData(trmt)
if (trmtExist) then $
  void = oTrmt->SetData([trmt,newTrmt],/no_copy,/no_notify) ; modify treatment


void = oErr->SetData(error,/no_copy,/no_notify)
void = oDat->SetData(data,/no_copy,/no_notify)
void = oind->SetData(vout,/no_copy,/no_notify)
 
; Update 'OUTPUTREF' entry
void = oCmds[1]->AddItem('FULLOUTPUTID',oRes->GetFullIdentifier(),/overwrite)

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DaveopDataConcatenate::ResetOpAttributes
; 
; PURPOSE:
;   Reset attributes for this operation to reflect the currently
;   selected datasets
;
; PARAMETERS:
;
; KEYWORDS:
;
;
function DaveopDataConcatenate::ResetOpAttributes, selNames
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the all datasets in Data Manager
oDatasets = oTool->GetDAVEDataManagerContents(count=nMax)
if (nMax lt 2) then return, 0

; Retrieve ids and names of datasets
self->GetProperty, types=validTypes ; should be ['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']
nValidTypes = n_elements(validTypes)
nValid = 0
For i=0,nmax-1 do begin
    if (~obj_valid(oDatasets[i])) then continue

    ;; Does dataset contain valid target data type?
    j = 0
    repeat begin
        oRes = oDatasets[i]->GetByType(validTypes[j],count=found)
    endrep until (found || ++j ge nValidTypes)

    if (~obj_valid(oRes)) then continue

    oDatasets[i]->GetProperty, name=name
    names = (n_elements(names) gt 0)? [names,name] : name
    validDatasets = (n_elements(validDatasets) gt 0)? [validDatasets,oDatasets[i]] : oDatasets[i]
    
    nValid++
endfor

if (nValid lt 2) then return, 0

; number of valid datasets should not exceed hardnmax
if (nValid gt Self.hardNmax) then begin
   msg = "Maximum datasets that can be concatenation is "+string(Self.hardNmax,format='(I-3)')
   msg = [msg,'Consider deleting a few datasets fron the Data Browser']
   msg = [msg,'until the total number falls below '+string(Self.hardNmax,format='(I-3)')]
   Self->ErrorMessage, msg, severity=2
   return, 0
endif

self->SetProperty, nmax = nValid ; the nos of datasets with valid targets
if (ptr_valid(self.validDatasetsPtr)) then ptr_free, self.validDatasetsPtr
self.validDatasetsPtr = ptr_new(validDatasets) ; the object refs of valid datasets found in DM


; Set -
; ndata to number of selected dataset
; range of ndata to between 2 and nos of valid datasets
; a default name to be used to store the result
self->SetPropertyattribute, 'ndata', valid_range=[2,self.nmax]
self->SetProperty, ndata=self.selCount

guess1 = self.resultName
guess2 = IDLitGetUniqueName(names, guess1) 
if (guess2 ne guess1) then begin
    pos = strpos(guess2,'_',/reverse_search)
    if (pos ne -1) then begin
        guess1  = strmid(guess2,0,pos) + ' '
        len = strlen(guess2) - (pos+1)
        guess1 += strmid(guess2,pos+1,len)
    endif
endif

self.namechanged = 0            ; indicate using default result name
self.resultName = guess2        ; may contain an underscore
self.resultID = guess1          ; the underscore replaced by space

if (ptr_valid(self.namesPtr)) then ptr_free, self.namesPtr
self.namesPtr = ptr_new(names)  ; the string names associated with the valid datasets found in the DM

; Assign the valid datasets (upto nmax) to the dataset* (registered) properties of the operation
for i = 0,self.nmax-1 do begin
    propName = Self.propertyNames[i]
    self->SetPropertyAttribute, propName, enumlist=(names)
    index = i
    if (i lt self.selCount) then begin
        ;; assign selected datasets as defaults
        index = where(names eq selNames[i], cnt)
        index = (cnt eq 1)? index[0] : i
    endif
    self.datasetsIndex[i] = index
endfor

return, 1

end


;===============================================================================
; DaveopDataConcatenate::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Concatenate the specified datasets 
;   and store the results in a new dataset stored in the Data Browser.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DaveopDataConcatenate::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt lt 2) then begin
    oTool->StatusMessage, 'Please, select at least two top-level datasets from the Data Browser tree.'
    return, obj_new()
endif

; Locate datasets containing experiment data
; Only top-level entries in the Data Manager Browser can be valid datasets
self->GetProperty, types=validTypes ; should be ['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']
nValid = n_elements(validTypes)
validName = 'DAVE Data Manager'
for i = 0,n_elements(oSelections)-1 do begin
    ;; Only top-level dataset selections from the data browser can be processed
    ;; ie the this operation is design to deal with complete datasets
    oTop = getmaindataobjectparent(oSelections[i])
    if (~obj_valid(oTop)) then continue
    if (oTop ne oSelections[i]) then continue ; ie selection must be a main entry in the tree

    ;; Search for one of the valid data types from this selection.
    ;; Essentially, the valid types refer to the experimental or plottable data containing
    ;; an independent, dependent and error component!
    j = 0
    repeat begin
        oRes = oSelections[i]->GetByType(validTypes[j],count=found)
    endrep until (found || ++j ge nValid)


    if (~obj_valid(oRes)) then continue
    oSelections[i]->GetProperty, name=selName
    selNames = (n_elements(selNames) gt 0)? [selNames,selName] : selName
endfor

if (n_elements(selNames) eq 0) then begin
    oTool->StatusMessage, 'Concatenation cannot be performed using selected datasets!'
    return, obj_new()
endif

self.selCount = n_elements(selNames)


; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

;; Set operation properties based on the selected data
if (~self->ResetOpAttributes(selNames)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
;hasPropSet = 0b
if (doUI) then begin
    ;; Record initial properties for this operation.
    if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
endif

;; Perform Concatenation and record changes in undo/redo buffer
if (~self->Concatenate(oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DaveopDataConcatenate::Concatenate
; 
; PURPOSE:
;   Perform difference b/n two datasets and record changes in the
;   operation's undo/redo buffer.
;
; PARAMETERS:
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0
;
function DaveopDataConcatenate::Concatenate, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; create a command object to store useful info. The target for the
; command is the operation itself. For the concatenate operation, only
; one concatenate will be performed per command set object created! If
; this were not the case, then it would be better to make a dataset
; object as the target.
oCmd = obj_new('IDLitCommand',target_identifier=self->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0
oCmdSet->Add, oCmd              ; Add the command to command set

; Operation settings
;resName = IDLitGetUniqueName((*self.namesPtr),self.resultName)
resName = self.resultName
nData = self.nData
algoIndex = self.algorithm
algoName = (self.algorithms)[algoIndex]
;targetIDs = (*self.targetIDsPtr)[self.datasetsIndex[0:nData-1]]
;datasetIDs = (*self.datasetIDsPtr)[self.datasetsIndex[0:nData-1]]
oDatasets = (*self.validDatasetsPtr)[self.datasetsIndex[0:nData-1]]
names = (*self.namesPtr)[self.datasetsIndex[0:nData-1]]

newTrmt = ['==================================================' $
            ,'Timestamp: '+systime()]
newTrmt = [newTrmt,'Concatenation operation performed using "'+algoName+'" algorithm']
idDatasets = strarr(nData)
pointer=0
for i=0,nData-1 do begin
   oDataset = oDatasets[i]
   idDatasets[i] = oDataset->GetFullIdentifier()

   if (~obj_valid(oDataset)) then continue

   oDataset->GetProperty, dataRef=oDat, errorRef=oErr, axis1Ref=oInd  ;get the dependent and error in dependent data
   
   if (~obj_valid(oDat) || ~obj_valid(oErr) || ~obj_valid(oInd)) then continue

   if (~oDat->GetData(dat) || ~oErr->GetData(err) || ~oInd->GetData(vin) ) then begin
       msg = 'Concatenate: Error retrieving data from dataset!'
       oTool->StatusMessage,msg
       return, 0
   endif

   if (i eq 0) then begin
       dS1 = size(dat)
       dS2 = size(err)
       dS3 = size(vin)
       ;; dims and nos of elements should be the same for data and error
       passed = ((dS1[0] eq dS2[0]) && (dS1[1] eq dS2[1]) && (dS1[2] eq dS2[2])) and $
                ((dS3[1] eq dS3[1]))
       if (~passed) then begin
           msg = 'Concatenate: Inconsistent dimensions and/or nos of elements!'
           oTool->StatusMessage,msg
           return, 0
       endif
       firstDataset = oDataset
   endif
   s1 = size(dat)
   s2 = size(err)
   s3 = size(vin)
   ;; dims and nos of elements should be the same for data and error
   passed = ((s1[0] eq s2[0]) && (s1[1] eq s2[1]) && (s1[2] eq s2[2])) and  $
            ( (s1[1] eq s3[1]) && s1[0] eq dS1[0] && s2[0] eq dS2[0] && s3[0] eq dS3[0]  ) 
    if (~passed) then begin
       msg = 'Concatenate: Inconsistent dimensions and/or nos of elements!'
       oTool->StatusMessage,msg
       return, 0
   endif
 ;

if i eq 0 then begin
data=dat & error=err & vout=vin & pointer=[pointer,dS3[1]]
endif else begin
data=[data,dat]
error=[error,err]
vout=[vout,vin]
pointer=[pointer,pointer[i]+S3[1]]
endelse


   newTrmt = [newTrmt,'Include "'+names[i]+'" in concatenation']
endfor

case strupcase(algoName) of
   'AUTO': begin         ;simple Concatenation
 order=sort(vout)
 data=data[order,*]
 error=error[order,*]
 vout=vout[order]
     end
 
   'USER INPUT': begin

;the followowing function allows the user to cull the consistuent data sets
Concatenate_Structure=WD_ConcatenateUserInput(data,error,vout,pointer)
;print,'Concatenate_Structure.success ', Concatenate_Structure.success
     if (~Concatenate_Structure.success) then begin
       msg = 'User Assisted Concatenation has been Cancelled'
       oTool->StatusMessage,msg
       return, 0
   end
 
data=Concatenate_Structure.data
error=Concatenate_Structure.error
vout=Concatenate_Structure.vout
if (ptr_valid(self.Concatenate_StructurePtr)) then ptr_free, self.Concatenate_StructurePtr
self.Concatenate_StructurePtr = ptr_new(Concatenate_Structure) ; this will be used in the redo operation


   end

   else: begin
       oTool->StatusMessage, 'Concatenation could not be performed - unknown algorithm!'
       return, 0
   end
endcase

; Store useful details
void = oCmd->AddItem('IDDATASETS',idDatasets)
void = oCmd->AddItem('DATASETNAMES',names)
void = oCmd->AddItem('ALGORITHMNAME',algoName)

; copy of the first dataset into the results dataset
; NB this creates an entry in the data browser
resID = (self.namechanged)? resName : self.resultID
void = oCmd->AddItem('OUTPUTNAME',resName)
void = oCmd->AddItem('OUTPUTID',resID)
if (obj_hasmethod(firstDataset,'Clone')) then begin
   ; use clone method to replicate first dataset
   oRes = firstDataset->Clone()
   oRes->SetProperty, name=resName, identifier=resID
   oTool->AddDatasetToDataManager, oRes
endif else begin
   ; use DM to replicate first dataset
   oDM = oTool->GetService("DAVEDATA_MANAGER")  ;; get DM service
   if (~oDM->CopyData(firstDataset->GetFullIdentifier(), name=resName, copyid=resID)) then return, 0
   oRes = oTool->GetByIdentifier(resID)
endelse

; Get the treatment info and update it
trmtExist = 0
oRes->GetProperty, trmtRef=oTrmt, dataRef=oDat, errorRef=oErr, axis1Ref=oInd
if (obj_valid(oTrmt)) then $
  trmtExist = oTrmt->GetData(trmt)
if (trmtExist) then $
  void = oTrmt->SetData([trmt,newTrmt],/no_copy,/no_notify) ; modify treatment

void = oErr->SetData(error,/no_copy,/no_notify)
void = oDat->SetData(data,/no_copy,/no_notify)
void = oind->SetData(vout,/no_copy,/no_notify)
 
; Save useful info for redo/undo
;void = oCmd->AddItem('FIRSTDATASETID',firstDatasetID)
void = oCmd->AddItem('FULLOUTPUTID',oRes->GetFullIdentifier())

; No need to notify observers because there should be none!
;oDat->notifyDataChange
;oDat->notifyDataComplete

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; DaveopDataConcatenate::Cleanup
; 
; PURPOSE:
;   DaveopDataConcatenate class cleanup
;
pro DaveopDataConcatenate::Cleanup
compile_opt idl2

if (ptr_valid(self.validDatasetsPtr)) then ptr_free, self.validDatasetsPtr
if (ptr_valid(self.namesPtr)) then ptr_free, self.namesPtr
if (ptr_valid(self.Concatenate_StructurePtr)) then ptr_free, self.Concatenate_StructurePtr

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DaveopDataConcatenate::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DaveopDataConcatenate::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is reversible
; This operation is not expensive
if (~self->IDLitOperation::Init(NAME='Data Concatenate' $
                                ,reversible_operation=1,expensive_computation=0 $
                                ,_EXTRA=etc)) then return, 0
;types=['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register properties for this operation
self.algorithms = ['AUTO','USER INPUT']
self.algorithm = 0
self->RegisterProperty, 'algorithm', description='The algorithm to be used for concatenation' $
  ,name='Algorithm to use',sensitive=1,enumlist=self.algorithms
self->RegisterProperty, 'resultName',/string, description='Name of dataset to store result' $
  ,name='Output dataset name',sensitive=1
self->RegisterProperty, 'ndata', /integer, description='Number of datasets to be concatenated' $
  ,name='Nos of datasets to concatenate',sensitive=1,valid_range=[2,2]

Self.hardNmax = 99
for i = 0,Self.hardNmax-1 do begin
    self.datasetsIndex[i] = i
    NicePropName = 'Dataset '+strtrim(string(i+1),2)
    prop = 'dataset'+strtrim(string(i+1, format='(I02)'),2)
    self->RegisterProperty, prop, enumlist=['empty'], description=NicePropName $
      ,name=NicePropName, sensitive=1,hide=1
    Self.propertyNames[i] = prop
endfor

self.resultName = 'ConcatenatedResult'
; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DaveopDataConcatenate__define
; 
; PURPOSE:
;   DaveopDataConcatenate class structure definition
;
pro DaveopDataConcatenate__define

compile_opt idl2

struc = {DaveopDataConcatenate $
         ,inherits IDLitOperation $
         ,hardNmax:99 $
         ,nmax:0B $             ; maximum nos of datasets available (present in data browser)
         ,datasetsIndex:bindgen(99) $ ; indices (of valid datasets) to be included in concatenation. 
                                             ;An absolute limit of 99 is imposed! Therefore nmax must be <= 99
         ,propertyNames:strarr(99) $   ; property names created dynamically to represent datasets
         ,namechanged:0B $       ; use to indicate when resultName field has been modified
         ,resultName:'' $       ; name of dataset to store the concatenation
         ,resultID:'' $         ; base id of dataset to store the concatenation
         ,algorithms:['',''] $  ; available algorithms to perform concatenation
         ,algorithm:0B  $       ; the one used
         ,ndata:0B $            ; nos of datasets to be included in concatenation
         ,selCount:0B $         ; nos of datasets currently selected
         ,validDatasetsPtr:ptr_new() $ ; ptr to ids of valid datasets
         ,namesPtr:ptr_new() $  ; ptr to names of valid datasets
         ,Concatenate_StructurePtr:ptr_new() $  ; ptr to User Defined Concatenated Data Sets
        }

end
