;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; NAME:
;  PAN_HydrogenRotRecoil
;
; PURPOSE:
;  PAN model function to determine hydrogen recoil scattering including 
;  rotational transitional terms for use in fitting data in the impulse 
;  approximation.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Timothy Prisk
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.

;+
; ARCSResWidth
;
; PURPOSE:
;   Evaluates the instrumental resolution width at any energy transfer
;   These are semi-emperical values determined for the ARCS instrument 
;   and then fitted to a polynomial of degree 4
;   W = b0 + b1*Et + b2*Et^2 + b3*Et^3 + b4*Et^4; 
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;   eTransfer:  the neutron energy transfer at which the resolution width is required
;
; RETURN VALUE:
;   the instrumental resolution width
;-
function ARCSResWidth, eTransfer, _Extra=extra
compile_opt idl2

if (n_elements(eTransfer) eq 0) then eTransfer=0.0

; parameters for ARCS instrument at Ei = 500 meV
b0 =  15.53747
b1 = -0.03454
b2 =  1.96388E-5
b3 = -2.38325E-8
b4 =  5.60666E-11

Return, b0 + b1*eTransfer + b2*eTransfer^2 + b3*eTransfer^3 + b4*eTransfer^4

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


function Pan_HydrogenRotRecoil,x,parms,$
  parmnames = parmnames, $
  canDraw = canDraw, $
  drawMessage = drawMessage, $
  initParms = initParms, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, $
  qvals=qvals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, xrange=xrange, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

compile_opt idl2

canDraw = 0
twoDimFlag = 0
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
                             ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                             ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if N_params() eq 0 then begin

  parmnames = ['KE', 'A1', 'A2', 'A3', 'A4', 'A5', 'A6', 'shift']
  initParms = [60.0,1.0,1.0,1.0,1.0,1.0,1.0,0.0]
  Return,-1
endif

; Parameter definition and initalization.
KE = parms[0]
A1 = parms[1]
A2 = parms[2]
A3 = parms[3]
A4 = parms[4]
A5 = parms[5]
A6 = parms[6]
shift = parms[7]

; Define constants.
hbarsquared = 4.18016 ; meV amu A^2
m  = 1.007825*2.0     ; amu
B0 = 7.3567           ; meV
D0 = 0.00569          ; meV
H0 = 6.4e-6           ; meV

; Convert kinetic energy to peak width.
KE_meV = KE/11.6
sigmaK = Sqrt(KE_meV*(2.0*m)/(3.0*hbarsquared))
sigmaS = (hbarsquared/m)*Qgroup*sigmaK

; Determine peak positions.
Recoil = 0.5*(hbarsquared/m)*Qgroup^2.0

Rot1 = B0*1.0*(1.0 + 1.0) - D0*(1.0*(1.0 + 1.0))^2.0 + H0*(1.0*(1.0 + 1.0))^3.0
Rot2 = B0*2.0*(2.0 + 1.0) - D0*(2.0*(2.0 + 1.0))^2.0 + H0*(2.0*(2.0 + 1.0))^3.0
Rot3 = B0*3.0*(3.0 + 1.0) - D0*(3.0*(3.0 + 1.0))^2.0 + H0*(3.0*(3.0 + 1.0))^3.0
Rot4 = B0*4.0*(4.0 + 1.0) - D0*(4.0*(4.0 + 1.0))^2.0 + H0*(4.0*(4.0 + 1.0))^3.0
Rot5 = B0*5.0*(5.0 + 1.0) - D0*(5.0*(5.0 + 1.0))^2.0 + H0*(5.0*(5.0 + 1.0))^3.0
Rot6 = B0*6.0*(6.0 + 1.0) - D0*(6.0*(6.0 + 1.0))^2.0 + H0*(6.0*(6.0 + 1.0))^3.0

C1 = Recoil + Rot1 - shift
C2 = Recoil + Rot2 - shift
C3 = Recoil + Rot3 - shift
C4 = Recoil + Rot4 - shift
C5 = Recoil + Rot5 - shift
C6 = Recoil + Rot6 - shift

; Calculate resolution at peak center.
ResWidths = ARCSResWidth([C1,C2,C3,C4,C5,C6])

; Observed with is intrinsic width and res width added in quadrature .
StoW = Sqrt(8.0*Alog(2.0))
sigmaO = sqrt((ResWidths/StoW)^2 + sigmaS^2)

; The model is a sum of six peaks.
peak1 = (A1/sqrt(2.0*!dpi*sigmaO[0]^2))*exp(-0.5*((x-C1)/sigmaO[0])^2)
peak2 = (A2/Sqrt(2.0*!dpi*sigmaO[1]^2))*Exp(-0.5*((x-C2)/sigmaO[1])^2)
peak3 = (A3/Sqrt(2.0*!dpi*sigmaO[2]^2))*Exp(-0.5*((x-C3)/sigmaO[2])^2)
peak4 = (A4/Sqrt(2.0*!dpi*sigmaO[3]^2))*Exp(-0.5*((x-C4)/sigmaO[3])^2)
peak5 = (A5/Sqrt(2.0*!dpi*sigmaO[4]^2))*Exp(-0.5*((x-C5)/sigmaO[4])^2)
peak6 = (A6/Sqrt(2.0*!dpi*sigmaO[5]^2))*Exp(-0.5*((x-C6)/sigmaO[5])^2)

return, peak1 + peak2 + peak3 + peak4 + peak5 + peak6

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


pro TestPan_HydrogenRotRecoil, yout=yout, plot=plot
  compile_opt idl2

  parms = [10.0,0.5,0.1,0.2,3.0,500.0,0.0,11.61,11.835,18.50,1.0,0.01]
  x = Findgen(301)*2.0 - 50.0

  void = Pan_HydrogenRotRecoil(parmnames=pnames)
  yout = Pan_HydrogenRotRecoil(x,parms,qgroup=10.0)
  if (Keyword_set(plot)) then begin
    title = 'Hydrogen Recoil Peaks'
    p = Plot(x,yout,title=title,symbol=0,linestyle=0)
  endif

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
