; $Id$
;###############################################################################
;
; NAME:
;  PAN_RandomJumpDiffusion_2D
;
; PURPOSE:
;  Implementation of the random jump diffusion model for use in fitting
;  data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or anq other
;  characteristic. The use of certain trade names or commercial
;  products does not imply anq endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function PAN_RandomJumpDiffusion_2D,x,parms,                       $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 1   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0           ; does function require external convolution? 0=>No, 1=>Yes
                            ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                            ; Also set to No, if you DO NOT wish anq external convolution to be performed at all 

nsingle = 5
nmulti = 3
if (N_params() eq 0) or (N_params() eq 1) then begin

  ; Define hbar according to the energy units
  eUnits = xunits.Toupper()
  eUnitsKnown = eUnits.Equals('MEV') || eUnits.Equals('UEV')
  if (eUnitsKnown) then begin
    if (eUnits.Equals('UEV')) then hbar = 658.21        ; ueV.ps
    if (eUnits.Equals('MEV')) then hbar = 0.65821       ; meV.ns
  endif else begin
    ; Unknown energy units so let user specify
    msg = 'The unit of energy for the data could not be determined!'
    msg = [msg,'Is the unit of the energy axis in ueV or meV?',' ']
    msg = [msg,'Click Yes if energy unit is ueV',' ']
    msg = [msg,'Or No if energy unit is meV']
    title = 'Random Jump Diffussion 2D -- Specify Energy Unit'
    if (N_elements(wTLB) eq 0) then wTLB=0L
    decision = Dialog_message(msg,title=title,/question,dialog_parent=wTLB)
    xunits = (decision.Equals('Yes'))? 'ueV' : 'meV'
    hbar = (decision.Equals('Yes'))? 658.21 : 0.65821
    eUnits = xunits.Toupper()
  endelse
  if (Isa(func_dataHash,'hash')) then func_dataHash['hbar'] = hbar

  tUnits = eUnits.Equals('UEV')? 'ps' : 'ns'     ; time unit depends on energy unit of data

  parmnames = ['RMS Vib. Displacement (Ang)',          $
    'Bond Length (Ang)',                           $
    'Trans Diffusivity (Ang^2/'+tUnits+')',        $
    'Site Residence Time ('+tUnits+')',                  $
    'Rot. Correlation Time ('+tUnits+')',          $
    'Intensity'+Strtrim(String(Indgen(nq)),2),  $
    'Center'+Strtrim(String(Indgen(nq)),2),     $
    'EISF'+Strtrim(String(Indgen(nq)),2)       ]
  single_parmnames = parmnames[0:nsingle-1]
  multi_parmnames = ['Intensity','Center','EISF']

  if Arg_present(initParms) then begin
    intensity = max(yVals,dimension = 1,index)
    index = index - indgen(nq)*nx ; index contains the 1D subscripts so need to reform
    center = xvals[index]

    ; assuming a single main peak then estimate HWHM for the _largest_ Q value
    hm = 0.5*intensity[nq-1]
    res = Where(yVals[*,nq-1] ge hm, found)
    if (found gt 0) then begin
      Gamma = 0.5*xvals[res[found-1]] - xvals[res[0]]
    endif
    
    tau0 = 0.25                               ; according to Sankar Nair, this is the approximate
                                              ; residence time for bulk water at room temperature.

    D = Gamma/(1 - tau0*Gamma*qVals[nq-1]^2)  ; from defn of Gamma
    U = 0.0
    tau = 0.1
    b = 1.0                                   ; initial guess for bond length
    eisf = 0.5+Dblarr(nq)
    initParms = [U,b,d,tau0,tau,intensity,center,Eisf]
  endif
  Return,-1
endif

if (~Isa(resPtr,'Pointer')) then Return, Fltarr(nx,nq)*0.0 ; can't proceed without a pointer to the resolution function
yout = Fltarr(nx,nq)
yres = (*resPtr).data
if (yres.ndim ne 2) then Return, yout   ; bail if resolution is not 2D

hbar = 658.21  ; ueV.ps; assumes energy units in microvolts! This is a precaution in case hbar defn was not stored in the hash earlier
if (Isa(func_dataHash,'hash') && func_dataHash.Haskey('hbar')) then hbar = func_dataHash['hbar']


index = indgen(nq)+nSingle
U = parms[0]
b = parms[1]
D = parms[2]
tau0 = parms[3]
tau = parms[4]
intensity = ux#(parms[index])
center = ux#parms[index+nq]
eisf = ux#parms[index+2*nq]

jo = ux#(Sph_bessel(qVals*b,0))
j1 = ux#(Sph_bessel(qVals*b,1))
j2 = ux#(Sph_bessel(qVals*b,2))
DQsq = D*qVals^2
gamma = DQsq/(1.0+tau0*DQsq)
gamma = ux#gamma
dwf_term = ux#(Exp(-(qVals*U)^2/3.0))

; delta function term
; area is eisf located at center
pseudo_delta = Fltarr(2,nq)             ; delta-function parameters
pseudo_delta[0,*] = parms[index+2*nq]   ; delta area
pseudo_delta[1,*] = parms[index+nq]     ; delta position

; Spherical bessel Lorentzian terms
xq = x#uq
x_cen = xq-center
term1 = (jo^2)*hbar*gamma/((hbar*gamma)^2+x_cen^2)
term2 = 3.0*(j1^2)*hbar*(gamma+1.0/(3.0*tau))/((hbar*(gamma+1.0/(3.0*tau)))^2+x_cen^2)
term3 = 5.0*(j2^2)*hbar*(gamma+1.0/(1.0*tau))/((hbar*(gamma+1.0/(1.0*tau)))^2+x_cen^2)
sqe_rjd = (1.0/!dpi)*(term1+term2+term3)

; Perform convolution with resolution
; sqe_rjd is convoluted seperatedly as well as the delta and
; a sum of the two is returned by pan_convolute()
yout = (intensity*dwf_term)*Pan_convolute(x,resPtr,(1-eisf)*sqe_rjd,pseudo_delta = pseudo_delta)

Return,yout

end
