; $Id$
;###############################################################################
;
; NAME:
;  PAN_QENS
;
; PURPOSE:
;  QENS function for use in fitting data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function pan_qens1_2D,x,parms,$
                    	parmnames = parmnames, $
                    	canDraw = canDraw, $
                    	drawMessage = drawMessage, $
                    	func_dataHash=func_dataHash, initParms=initParms, twoDimFlag=twoDimFlag, $
                    	qvals=qVals,xvals=xVals,yvals=yVals, evals=eVals, qGroup=qVal, wTLB=wTLB, $
                      resPtr = resPtr, resolutionRequiredFlag=resolutionRequiredFlag, resLimit=rLimit, $
                      _Extra = extra

canDraw = 0
resolutionRequiredFlag = 1   ; ==> implies this function requires a resolution function to be defined
nq = N_elements(qVals)

if n_params() eq 0 then begin
  twoDimFlag = 1
  ; Evalute areas
  parmnames = ['FWHM0']
  aname = 'DeltaArea'
  bname = 'LorArea'
  cname = 'Center'
  for i = 0,nq-1 do begin
    parmnames = [parmnames,aname+strtrim(string(i+1),2)]
    parmnames = [parmnames,bname+Strtrim(String(i+1),2)]
    parmnames = [parmnames,cname+Strtrim(String(i+1),2)]
  endfor
  
  initparms = Fltarr(3*nq+1)

  ; generic guess for initial parameters
  initparms[0] = 0.5
  initparms[3*Indgen(nq) + 1] = 0.2    ; the delta areas
  initparms[3*Indgen(nq) + 2] = 0.8    ; the Lorentzian areas
  initparms[3*Indgen(nq) + 3] = 0.0    ; the Lorentzian/Delta positions (tied)

  if (Isa(func_dataHash,'hash')) then begin
    ; let user specify the expected Q dependence
    msg = ['Please indicate expected Q-Dependence for the Lorentzian component',' ']
    msg = [msg,'Click Yes for a Q-square Lorentzian width',' ']
    msg = [msg,'Or No for a Q-independent Lorentzian width']
    title = 'QENS 2D -- Specify Q-Dependence'
    if (N_elements(wTLB) eq 0) then wTLB=0L
    decision = Dialog_message(msg,title=title,/question,dialog_parent=wTLB)
    qDep = (decision.Equals('Yes'))? 2 : 0
    func_dataHash['initialized'] = 1
    func_dataHash['qdependence'] = qDep

    ; if we have data then 
    if (isa(xVals) && isa(yVals) ) then begin
      if (yVals.ndim eq 1) then begin
        ymin = yVals.Sort()
        bg = (N_elements(ymin) gt 5)? Mean(ymin[0:4]) : ymin[0]
        xpeaks = Get_peak_pos(xVals,yVals-bg,1,fwhm = fwhm)
        gam = 0.5*fwhm
        area = (Max(dat)-bg)*!pi*gam

        ; assisn 20% (arbitrary) of area to delta and he rest to the Lor.
        initparms[0] = fwhm / (qvals[groupNumber-1]^qDep)
        initparms[3*indgen(nq) + 1] = 0.25*area    ; the delta areas
        initparms[3*indgen(nq) + 2] = 0.25*area    ; the Lorentzian areas
        initparms[3*Indgen(nq) + 3] = xpeaks[0]   ; the Lorentzian/Delta positions (tied)
      endif else begin
        for i = 0,nq-1 do begin
          ymin = yVals[*,i].Sort()
          bg = (N_elements(ymin) gt 5)? Mean(ymin[0:4]) : ymin[0]
          xpeaks = Get_peak_pos(xVals,yVals[*,i]-bg,1,fwhm = fwhm)
          if (i eq 0) then initparms[0] = fwhm / (qvals[i]^qDep)
          gam = 0.5*fwhm
          area = (Max(yVals[*,i])-bg)*!pi*gam
          initparms[3*i + 1] = 0.25*area    ; the delta area
          initparms[3*i + 2] = 0.25*area    ; the Lorentzian area
          initparms[3*i + 3] = xpeaks[0]   ; the Lorentzian/Delta position (tied)
        endfor
      endelse
    endif
  endif

  return,-1
endif

nx = N_elements(x)
if (~isa(resPtr,'Pointer')) then return, fltarr(nx,nq)*0.0 ; can't proceed without a pointer to the resolution function

yout = fltarr(nx,nq)
qDep = func_dataHash['qdependence']
localResPtr = ptr_new({x:(*resPtr).x,data:(*resPtr).data[*,0],xdat_equals_xres:(*resPtr).xdat_equals_xres})
for i=0, nq-1 do begin
  fwhm = parms[0] * qvals[i]^qDep ; scale the global FWHM by the expected q-dependence
  gamma = 0.5*fwhm
  aDelta = parms[3*i+1]
  aLor   = parms[3*i+2]
  cen    = parms[3*i+3]
  Lor = aLor*(gamma/!dpi)/((x-cen)^2+gamma^2)   ; Lorentzian
  (*localResPtr).data = (*resPtr).data[*,i]
  yout[*,i] = pan_convolute(x,localResPtr,Lor,pseudo_delta = [[aDelta,cen]])
endfor

return,yout
end
