; $Id$
;###############################################################################
;
; NAME:
;  PAN_JumpDiffusion_SingwiSjolander_2D
;
; PURPOSE:
;  QENS - Jump Diffusion - Singwi-Sjolander model
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; QENS - Jump Diffusion - Singwi-Sjolander
function PAN_JumpDiffusion_SingwiSjolander_2D,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 1
twoDimFlag = 1
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 1           ; does function require external convolution? 0=>No, 1=>Yes
                            ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                            ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if N_params() eq 0 then begin
  twoDimFlag = 1
  ; Evalute areas
  parmnames = ['Tau','D']
  aname = 'Area'
  bname = 'Center'
  parmnames = [parmnames,aname+strtrim(string(indgen(nq)),2)]
  parmnames = [parmnames,bname+Strtrim(String(Indgen(nq)),2)]
  single_parmnames = ['Tau','D']
  multi_parmnames = ['Area','Center']
  
  ; Define hbar according to the energy units
  if (Isa(func_dataHash,'hash')) then begin
    eUnits = xunits.Toupper()
    print,'############ xunits = ',xunits
    eUnitsKnown = eUnits.Equals('MEV') || eUnits.Equals('UEV')
    if (eUnitsKnown) then begin
      if (eUnits.Equals('UEV')) then hbar = 658.0
      if (eUnits.Equals('MEV')) then hbar = 0.658
    endif else begin
      ; Unknown energy units so let user specify
      msg = 'The unit of energy for the data could not be determined!'
      msg = [msg,'Is the unit of the energy axis in ueV or meV?',' ']
      msg = [msg,'Click Yes if energy unit is ueV',' ']
      msg = [msg,'Or No if energy unit is meV']
      title = 'Singwi-Sjolander (Jump Diffusion) 2D -- Specify Energy Unit'
      if (N_elements(wTLB) eq 0) then wTLB=0L
      decision = Dialog_message(msg,title=title,/question,dialog_parent=wTLB)
      xunits = (decision.Equals('Yes'))? 'ueV' : 'meV'
      hbar = (decision.Equals('Yes'))? 658.0 : 0.658
    endelse
    func_dataHash['hbar'] = hbar
  endif
  
  Return,-1
endif
drawMessage = Strarr(4)
drawMessage[0:1] = ['Hold left mouse button down','and move to set function height and center']
drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

hbar = 658.0  ; ueV.ps [microVolts . picoseconds]
              ; default to energy units in microvolts in case it was not set earlier!
if (Isa(func_dataHash,'hash') && func_dataHash.haskey('hbar')) then hbar = func_dataHash['hbar']

if (Keyword_set(changefirst)) then begin
  ; user is initializing the function graphically
  ; this block of code is executed when the mouse is first release at the location for the function centroid
  ; translate the position into suitable parameters.
  ;Print,'###Changefirst called within function...'
  func_dataHash['ymax'] = yMouseClick     ; user selected max height
  fwhm = 0.15*(xrange[1]-xrange[0])       ;  assign an arbitrary FHHM that is 15% of the xrange
  gam = 0.5*fwhm
  cen = xMouseClick
  area = yMouseClick*gam*!dpi
  help,'changefirst, cen',cen
  qVal   = qVals[groupNumber-1]

  ; According to Singwi-Sjolander model, the width of the Lorentzian fit to the data is defined as
  ; gamma = hbar.D.Q^2/(1 + D.tau.Q^2)
  ; ==> tau = hbar/gamma - 1/(D.Q^2)
  D = 2.0                                     ; arbitrary assign diffusion constant in 10^(-9) m^2/s  
  tau = hbar/gam - 1.0/(D*qval^2)             ; and calculate tau
  
  parms[0] = tau
  parms[1] = D
  gamQ = hbar*D*qvals^2/(1 + D*tau*qvals^2)  ; gamma as function of Q
  areaIndex = indgen(nq)+2
  cenIndex = indgen(nq)+2 + nq
  parms[areaIndex] = yMouseClick*gamQ*!dpi       ; area as function of Q
  parms[cenIndex] = cen
endif

if (Keyword_set(changesecond)) then begin
  ; user is initializing the function graphically
  ; this block of code is executed when the mouse is move to define the function width
  ; translate the position into suitable parameters.
  ;Print,'###Changesecond called within function...'
  cen    = parms[(groupNumber-1)+2 + nq]          ; the initial center determined from changefirst above
  gam    = Abs(xMouseClick - cen)
  area   = func_dataHash['ymax']*gam*!dpi    ; adjust area since gamma has changed - use the previously selected max height
  qVal   = qVals[groupNumber-1]
  
  ; According to Singwi-Sjolander model, the width of the Lorentzian fit to the data is defined as
  ; gamma = hbar.D.Q^2/(1 + D.tau.Q^2)
  ; ==> tau = hbar/gamma - 1/(D.Q^2)
  D = 2.0                                     ; arbitrary assign diffusion constant in 10^(-9) m^2/s  
  tau = hbar/gam - 1.0/(D*qval^2)             ; and calculate tau
  
  parms[0] = tau
  parms[1] = D
  gamQ = hbar*D*qvals^2/(1 + D*tau*qvals^2)   ; gamma as function of Q
  areaIndex = Indgen(nq)+2
  cenIndex = Indgen(nq)+2 + nq
  parms[areaIndex] = func_dataHash['ymax']*gamQ*!dpi       ; area as function of Q
  ; cen are unchanged
endif

tau = parms[0]
D   = parms[1]
areaIndex = indgen(nq)+2
cenIndex = indgen(nq)+2 + nq
area = parms[areaIndex]
cen = parms[cenIndex]
gamQ = ux # (hbar*D*qvals^2/(1 + D*tau*qvals^2))          ; gamma as function of Q and expanded along x
yout = (ux#area)/!pi*gamQ/(gamQ^2 + ((x#uq) - (ux#cen))^2)

Return,yout
end