;==============================================================================
; Cleanup pro
pro ng5CalibDataA4::cleanup
compile_opt idl2

ptr_free, self.pData,self.pA4p,self.pA4,self.pSen

end

;==============================================================================
; Read calibration data file
function ng5CalibDataA4::readFile
compile_opt idl2

;begin error handler------------------------------------------------------------
defsysv,'!debug',exist=debugExist
if (debugExist && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'Error encountered in ng5CalibDataA4::isNG5Calib()'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle)
        free_lun,lun,/force ; close file, if open
        catch, /cancel
        return, 0
    endif
endif
;end error handler-------------------------------------------------------------

filename = self.filename
openr, lun,filename,/get_lun
buffer = strarr(14)
readf, lun, buffer

;nDat = fix(strmid(buffer[0],65,4))
;nDat = fix((strsplit(buffer[0],' ',/extract))[7])
toks = strsplit(buffer[0],' ',/extract,count=ntoks)
nDat = fix(toks[ntoks-2])
;Ef = float((strsplit(buffer[7],' ',/extract))[2])
nPix = fix((strsplit(buffer[12],' ',/extract))[2])

colTitles = strsplit(buffer[13],' ',/extract,count=nCol)
a4Index = (where(colTitles eq 'A4',found))[0]
if (~found) then begin
  print, 'A4 column not found'
  return, 0
endif

A4 = fltarr(nDat)
data = fltarr(nDat,nPix)
buffer1 = fltarr(nCol)
buffer2 = fltarr(nPix)
x = findgen(nPix)+1
i = 0

while not eof(lun) do begin
  readf,lun,buffer1,buffer2
  A4[i] = buffer1[a4Index]
  data[i,*] = buffer2
  
  ; identify peak intensities and keep rolling sum
  peak_loc = get_peak_pos(x,buffer2,1,fwhm=fwhm,indices=indices)
  if (indices gt 0 and fwhm gt 0.0) then begin
    peakInt = (n_elements(peakInt) gt 0)? [peakInt,buffer2[indices]] : buffer2[indices]
  endif

  i++
endwhile
free_lun,lun,/force

; Determine standard deviation of peak intensities
if (n_elements(peakInt) ge 2) then begin 
  self.sigmaInt = stddev(peakInt) 
  self.meanInt = mean(peakInt)
endif

if (i ne nDat) then begin
  A4 = A4[0:i-1]
  data = data[0:i-1,*]
endif

self.nDat = i
self.nPix = nPix
self.Ef = 0.0
self.pData = ptr_new(data,/no_copy)
self.pA4 = ptr_new(A4,/no_copy)

return, 1
end

;==============================================================================
; Determine if this is an NG5 (SPINS) data file with PSD used in a+ mode
; where pixels are integrated in the vertical direction.
function ng5CalibDataA4::isNG5CalibFile
compile_opt idl2

;begin error handler------------------------------------------------------------
defsysv,'!debug',exist=debugExist
if (debugExist && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'Error encountered in ng5CalibDataA4::isNG5Calib()'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle)
        catch, /cancel
        return, 0
    endif
endif
;end error handler-------------------------------------------------------------

filename = self.filename
openr, lun,filename,/get_lun
buffer = strarr(14)
readf, lun, buffer
free_lun,lun,/force

; It must be an ICP I-buffer file: line 1, col 37 should be letter I
if (~strcmp(strmid(buffer[0],36,1),'I')) then return, 0

; Is there a '#ASD' entry on line twelve? If so, this is a+ mode SPINS data file
if (~strcmp(strmid(strtrim(buffer[12],2),0,4),'#ASD')) then return, 0

;; Measurement must have been performed in fixed-final energy mode
;if (~strcmp(strmid(buffer[8],27,8),'EA fixed')) then return, 0

return, 1
end

;==============================================================================
pro ng5CalibDataA4::getProperty,Efix=Efix,nDat=nDat,nPix=nPix $
    ,A4vals=A4, A4p=A4p, sen=sen,data=data,meanInt=meanInt,sigmaInt=sigmaInt,filename=filename
compile_opt idl2

if (arg_present(filename)) then begin
  filename = self.filename
endif
if (arg_present(Efix)) then begin
  Efix = 0.0
endif
if (arg_present(nDat)) then begin
  nDat = self.nDat
endif
if (arg_present(nPix)) then begin
  nPix = self.nPix
endif
if (arg_present(filename)) then begin
  filename = self.filename
endif
if (arg_present(meanInt)) then begin
  meanInt = self.meanInt
endif
if (arg_present(sigmaInt)) then begin
  sigmaInt = self.sigmaInt
endif
if (arg_present(A4)) then begin
  A4 = (ptr_valid(self.pA4))? (*self.pA4) : 0.0
endif
if (arg_present(A4p)) then begin
  A4p = (ptr_valid(self.pA4p))? (*self.pA4p) : 0.0
endif
if (arg_present(sen)) then begin
  sen = (ptr_valid(self.pSen))? (*self.pSen) : 0.0
endif
if (arg_present(data)) then begin
  data = (ptr_valid(self.pData))? (*self.pData) : 0.0
endif

end 

;==============================================================================
pro ng5CalibDataA4::setProperty,Ef=Ef,nDat=nDat,nPix=nPix $
    ,A4vals=A4, A4p=A4p, sen=sen,data=data
compile_opt idl2

if (n_elements(Ef) eq 1) then begin
  self.Ef = Ef
endif

if (n_elements(nDat) eq 1) then begin
  self.nDat = nDat
endif

if (n_elements(nPix) eq 1) then begin
  self.nPix = nPix
endif

if (n_elements(A4vals) gt 0) then begin
  if (ptr_valid(self.pA4)) then $
    (*self.pA4) = A4 $
  else $
    self.pA4 = ptr_new(A4)
endif

if (n_elements(A4p) gt 0) then begin
  if (ptr_valid(self.pA4p)) then $
    (*self.pA4p) = A4p $
  else $
    self.pA4p = ptr_new(A4p)  
endif

if (n_elements(sen) gt 0) then begin
  if (ptr_valid(self.pSen)) then $
    (*self.pSen) = sen $
  else $
    self.pSen = ptr_new(sen)
endif

if (n_elements(data) gt 0) then begin
  if (ptr_valid(self.pData)) then $
    (*self.pData) = data $
  else $
    self.pData = ptr_new(data)
endif

end

;==============================================================================
function ng5CalibDataA4::init, file, ftpObject=oFTP, fromFTP=fromFTP
compile_opt idl2

; filename parameter must be present
if (n_params() lt 1) then begin
  print, 'A calibration data filename is required'
  return, 0
endif

filename=file

fromFtp = (stregex(filename,'/pub/ncnrdata/',/fold,/bool))? 1 : 0
if (~fromFTP && ~file_test(filename)) then return, 0

if (keyword_set(fromFTP) && ~obj_valid(oFTP)) then return, 0

if (keyword_set(fromFTP)) then begin
  ; locate a suitable temporal directory
  tmpDir = !home_dir
  ; and copy file from the server into tmpfile on the host
  ; use this tmpfile below to load the data from
  tmpfile = tmpDir + path_sep() + 'NG5ftpData.txt' 
  filename = oFTP->GetFileContent(filename,localfilename=tmpfile)
  if (~file_test(filename,/read)) then return, 0  ; check local file exists and is readable
endif else begin
  ; check filename exist
  finfo = file_info(filename)
  if (finfo.read ne 1) then begin
    print,filename,' is not found or cannot be read'
    return, 0
  endif
endelse

self.filename = filename

; Is this a proper ng5 calibration file?
if (~self->isNg5CalibFile()) then begin
  print,file,' is not an appropriate SPINS calibration data file'
  return, 0
endif

; Read the data and init data structure
void = self->readFile()

if (fromFTP) then FILE_DELETE, filename  ;  delete temp data file
self.filename = file

return, 1
end

;==============================================================================
pro ng5CalibDataA4__Define
compile_opt idl2

struct = {ng5CalibDataA4 $
          ,filename:'' $
          ,Ef:0.0 $           ; Fixed analyzer final energy
          ,nDat:0 $           ; nos of data points
          ,nPix:0 $           ; nos of pixels
          ,meanInt:0.0 $      ; mean of peak intensities
          ,sigmaInt:0.0 $     ; std deviation of peak intensities
          ,pA4:ptr_new() $    ; A4 at each data point
          ,pA4p:ptr_new() $   ; Calculated A4 for each pixel
          ,pSen:ptr_new() $   ; Calc. sensitivity for each pixel
          ,pData:ptr_new() $  ; Calibration counts nDat x nPix
          } 
end