; $Id$
; ************************************ ;
; NAME:
;  DAVE_PARSE_MISSOURITAS
;
; PURPOSE:
;  This function parses a Missouri TAS data file and
;  returns the data contained therein in a structure
;  accessed via an output parameter DSTRUCT.  The input
;  parameter is DSTRING which is a string array containing
;  the entire contents of the data file.
;
; CATEGORY:
;  DAVE, Data Reduction, TAS.
;
; AUTHOR:
;   Richard Tumanjong Azuah
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
; ************************************ ;
function dave_parse_missouritas, dstring,dstruct    $
  ,header=header_string $
  ,headText = headText $
  ,dattaText = dattaText $
  ,indep_var_pos=indep_var_pos $
  ,dep_var_pos=dep_var_pos $
  ;;,plot_indep_var = plot_indep_var $
  ,lattice=lattice $
  ,fixedEType=fixedEType $
  ,fixedEValue=fixedEValue $
  ,dSpaceM=dSpaceM $
  ,dSpaceA=dSpaceA $
  ,orientation=orientation $
  ,sdesc=sdesc $
  ,anaPkgStr=anaPkgStr $
  ,errmsg = errmsg

errmsg = ''
catch,the_error
if the_error ne 0 then begin
   catch,/cancel
   errmsg = !error_state.msg
   return,0B
endif
nlines = n_elements(dstring)

header_counter = 0L
data_counter = 0L

; Go through each element of the string array and get out the first data line
; and the last data line
last_line_boolean = 0B
indep_index = -1
dep_index = -1
indep_var_pos = -1
dep_var_pos = -1
latt_index = -1
dsmon_index = -1
dsana_index = -1
fEi_index = -1
orient_index = -1
sdesc_index = -1
lattice = [0.0D,0.0D,0.0D,0.0D,0.0D,0.0D]
orientation = [0.0,0.0,0.0,0.0,0.0,0.0]
fixedEType = ''
fixedEValue = 0.0
dspaceM = 0.0
dspaceA = 0.0
sdesc = ''
anapkg = 0
doOrient = 0

headText = strarr(2,nlines)
header_string = []
ih = 0

for i = 0,nlines-1 do begin
   print,i
    if (strmid(dstring[i],0,1) ne '#') then continue ; don't analyze data lines
    
    toks = strtrim(strsplit(dstring[i],'=',/extract,count=ntoks),2)
    if (ntoks eq 2) then begin
       headText[0,ih] = toks[0]
       headText[1,ih] = toks[1] ;(ntoks ge 2)? strjoin(toks[1:ntoks-1],' ') : ' '
       header_string = [header_string,dstring[i]]
       ih++
    endif
    
    case strupcase(toks[0]) of
        '# COL_HEADERS': begin
            first_data_line = i+2
            toks = strsplit(dstring[i+1],' ',/extract,count=ntoks)
            titles = toks[1:ntoks-1]
            ncols = ntoks - 1
        end
;        '#COMMENT':
        '# DEF_X': indep_var_name = strtrim(toks[1],2)
        '# DEF_Y': dep_var_name = strtrim(toks[1],2)
;        '#SCAN': begin
;            indep_var_pos = fix(toks[1]) - 1
;            indep_index = (indep_var_pos lt 0)? -1 : 1 
;        end
;        '#SIGNAL': begin
;            dep_var_pos = fix(toks[1]) - 1 ; zero-based index
;            dep_index = (dep_var_pos lt 0)? -1 : 1 
;        end
        '# LATTICECONSTANTS': begin
            toks = strsplit(toks[1],',',/extract,count=ntoks)
            if (ntoks eq 6) then lattice =double(toks)
        end
;        '#FIXEDE': begin
;            if (ntoks eq 3) then begin
;                fixedEType = toks[1]
;                fixedEValue = float(toks[2])
;            endif
;        end
;        '#MONOSPACING': if (ntoks eq 2) then  dspaceM = float(toks[1])
;        '#ANASPACING': if (ntoks eq 2) then  dspaceA = float(toks[1])
;        '#ORIENT':   begin
;            if (ntoks eq 7) then orientation = float([toks[1],toks[2],toks[3],toks[4],toks[5],toks[6]])
;            lab1 = 'Orient'+strjoin([toks[1],toks[2],toks[3]])
;            lab2 = 'Orient'+strjoin([toks[4],toks[5],toks[6]])
;            doOrient = 1
;        end
        '# COMMAND':   if (ntoks eq 2) then sdesc = toks[1]
;        '#ANALYZERDETECTORMODE': begin
;            if (ntoks eq 3) then begin
;                anaPkg = 1
;                anaMode = fix(toks[1])
;                anaModeName = strtrim(toks[2],2)
;            endif
;        end
;        '#ANALYZERDETECTORDEVICESOFINTEREST': anaSigDets = toks[1:ntoks-1]
;        '#ANALYZERSDGROUP': anaSDDets = toks[1:ntoks-1]
;        '#ANALYZERDDGROUP': anaDDDets = toks[1:ntoks-1]
;        '#ANALYZERDOORDETECTORGROUP': anaTDDets = toks[1:ntoks-1]
;        '#ANALYZERPSDGROUP': anaPSDDets = toks[1:ntoks-1]
        
        else: begin
            ;; check for fit parameters present in some scans at end
            ;; of file
            if (strmatch(toks[0],'# Sum',/fold_case)) then $
              last_data_line = i-1
        end
    endcase

endfor
headText = headText[*,0:ih-1] ; exclude from #col_headers keyword and lower


; error message text sometimes embedded within the data block so quickly run through
; data block section and locate lines containing data
data_string = []
for i=first_data_line, nlines-1 do begin
   if (strmatch(strmid(dstring[i],0,1),'#',/fold_case)) then continue   ; exclude lines where first character begins with #
   data_string = [data_string, dstring[i]]
endfor 

ndata = n_elements(data_string)
if (ndata eq 0) then begin
  errmsg = 'Data block contains no data'
  return,0B
endif

data = fltarr(ncols,ndata)
dattaText = strarr(ncols,ndata)
for i = 0,ndata-1 do begin
    tmpBuf = strsplit(data_string[i],' ',/extract)
    dattaText[*,i] = tmpBuf[0:ncols-1]
;    tmpInd = where(tmpBuf eq 'N/A',indCnt)
;    if (indCnt gt 0) then tmpBuf[tmpInd] = 0.0 ;'-999'
    data[*,i] = float(tmpBuf[0:ncols-1])
endfor

; The indep variable
indep_var_pos = (where(titles eq indep_var_name, found))[0]
if (indep_var_pos lt 0) then begin
    ;; find first varying device
    i = -1
    found = -1
    small = 0.001
    while ((++i lt ncols) && (found eq -1)) do begin
        diff = abs(data[i,ndata-1] - data[i,0])
        colHeader = strupcase(titles[i])
        notAMon = colHeader ne 'MONITOR' 
        notADet = STRMID(colHeader,0,3) ne 'DET'
        notTime = colHeader ne 'TIME'
        found = ((diff gt small) && notAMon && notADet && notTime)? i : -1
    endwhile
    
    if (found gt -1) then begin
;        plot_indep_var = titles[found]
        indep_var_pos = found
    endif else begin
;        plot_indep_var = ''
        indep_var_pos = 0
    endelse
endif

; The dep variable. If the signal was not explicitly specified, use the detector
dep_var_pos = (where(titles eq dep_var_name, found))[0]
if (dep_var_pos lt 0) then begin
    dep_var_pos = where(strupcase(titles) eq 'DETECTOR')
    ;dep_var_pos = dep_var_pos[0] - 1 ; zero-based index
endif ;

;; Add an extra column
;; called 'Signal' and _copy_ the dependent data to it.
data = [data,data[dep_var_pos,*]]
dattaText = [dattaText,dattaText[dep_var_pos,*]]
titles = [titles,'Signal']
nCols++

; What is the signal error?
errIndex = where(strupcase(titles) eq 'DETECTORERR',foundErr)
if (foundErr) then begin
  error = reform(data[errIndex,*])
endif else begin
  error = reform(data[nCols-1,*]) ;reform(data[dep_var_pos,*])
  badInd = where(error le 0.0,cnt)
  if (cnt gt 0) then error[badInd] = 1.0
  error = sqrt(error)
endelse
; Create a structure for the data with the titles defined
; as the tag names.
; Note: TAS reduction code requires an INDEX column in the data structure so
; rename the first column from 'Pt.' to 'INDEX' in the data structure
; and to 'Data Point' in the titles variable
titles[0] = 'Data Point'
dstruct = create_struct('INDEX',ptr_new(reform(data[0,*])))
for i = 1,nCols-1 do dstruct = create_struct(dstruct,titles[i],ptr_new(reform(data[i,*])))

; Add an error column at the end
dattaText = [dattaText,transpose(string(error))]
dstruct = create_struct(dstruct,'ERROR',ptr_new(error,/no_copy))
titles = [titles,'ERROR']

; add the titles as the first row in the dattaText array
dattaText = transpose([transpose(titles), transpose(dattaText)])

return,1B
end
; ************************************ ;
pro test_read_missouritas, filename
read_ok = dave_read_filecontents(filename,dstring = dstring,errmsg = errmsg)
if ~read_ok then begin
   print,errmsg
   return
endif
parse_ok = dave_parse_missouritas(dstring,dstruct,header=header $
                          ,indep_var_pos=indep_ind $
                          ,dep_var_pos=dep_ind $
                          ,lattice=lattice $
                          ,fixedEType=fixedEType $
                          ,fixedEValue=fixedEValue $
                          ,dSpaceM=dSpaceM $
                          ,dSpaceA=dSpaceA $
                          ,orientation=orientation $
                          ,sdesc=sdesc $
                          ,anaPkgStr=anaPkgStr $
                          ,errmsg = errmsg)
if ~parse_ok then print,errmsg

if ~parse_ok then return

names = tag_names(dstruct)
x = (*dstruct.(indep_ind))
y = (*dstruct.(dep_ind))
dy = (*dstruct.error)
;plot,x,y,psym = 4
;errplot,x,y-dy,y+dy,width = 0.0
help,x,y

heap_free, dstruct
end
