;$Id$
;###############################################################################
;
; NAME:
;  SUBTRACT_SCAN
;
; PURPOSE:
;  See description below.
;
; CATEGORY:
;  DAVE, HFBS, data reduction
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;+
; NAME:
;       SUBTRACT_SCAN
;
; PURPOSE:
;
;   Simple function wrapper for the "subtract" functionality of HSCN_ADDRUN.
;   Suitable for invocation at the command line.  The function returns
;   a new object reference containing the data for the subtracted datasets.
;
; AUTHOR:
;
;       Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;       100 Bureau Drive
;   Gaithersburg, MD 20899
;       Phone: (301) 975-8135
;       E-mail: robert.dimeo@nist.gov
;       http://www.ncnr.nist.gov/staff/dimeo
;
; CATEGORY:
;
;       Data reduction, DAVE project, HFBS
;
; CALLING SEQUENCE:
;
;   OMERGE = SUBTRACT_SCAN(OARRAY)
;
; PARAMETERS:
;
;   OARRAY:   Array of 2 data objects to be subtracted (second from first).
;         These are assumed to be of the HSCN_DATA class.
;
; KEYWORDS:
;
;   None
;
; COMMON BLOCKS:
;
;   None
;
; REQUIREMENTS:
;
;   HSCN_ADDRUN.PRO
;   HSCN_DATA__DEFINE.PRO
;
; MODIFICATION HISTORY:
;
;       Written by Rob Dimeo, July 11, 2003
;-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function subtract_scan,oarray,interp = interp
; Check if correct number of parameters were passed in
if n_params() eq 0 then return,0
if n_elements(interp) eq 0 then interp = 0
ndata = n_elements(oarray)
if (ndata ne 2) then return,0

; Go through and make sure that the independent variables for each data set
; are set the same way.
ret = oarray[0]->get(indep_var = indep_var)
label = indep_var.label
for i = 1,ndata-1 do begin
   ret = oarray[i]->get(indep_var = indep_var)
   if indep_var.label ne label then begin
     str = 'ret = oarray[i]->set_indep_var(/'+strtrim(label,2)+')'
     r = execute(str,1)
   endif
endfor

for detector = 0,23 do begin
   for i = 0,ndata-1 do begin
     ret = oarray[i]->set(detector = detector)
     s = oarray[i]->create_ADDRUN_data_structure()
     if (i eq 0) then sarray = s else sarray = [sarray,s]
   endfor
   s_subtract = HSCN_ADDRUN(sarray,/subtract,interp = interp,msg = msg)
   heap_free,sarray
   if msg ne '' then begin
     void = dialog_message(msg,/error)
     return,0
   endif
   if detector eq 0 then s_subtract_array = s_subtract else $
     s_subtract_array = [s_subtract_array,s_subtract]
endfor
; Now cycle through the s_subtract_array, pull out the values and populate
; a new data array and all of the other relevant instance data for the
; HSCN_DATA class.
;
; Get the 1st element out
d = s_subtract_array[0]
x = *d.px
y = *d.py
dy = *d.pdy
mon = *d.pmon
dmon = *d.pdmon
heap_free,d
; Clone an existing data object.  ONEW will be the resulting data object
; of the add procedure.
ret = oarray[0]->clone(cloned_object = onew)
ret = onew->get(indep_var = indep_var)
*indep_var.pqty = x & nx = n_elements(x)
ret = onew->set(indep_var = indep_var)
data = fltarr(24,nx)
error = fltarr(24,nx)
data[0,*] = mon[*]
error[0,*] = dmon[*]

; Which independent variable is it?
case indep_var.identity of
'TCONTROL': $
   begin
     tcontrol = (tsample = x)
     time = findgen(n_elements(x))
   end
'TSAMPLE':  $
   begin
     tcontrol = (tsample = x)
     time = findgen(n_elements(x))
   end
'TIME':  $
   begin
     time = x
     tsample = (tcontrol = findgen(n_elements(x)))
   end
else:
endcase

for i = 1,23 do begin
   d = s_subtract_array[i]
   x = *d.px & y = *d.py & dy = *d.pdy
   mon = *d.pmon & dmon = *d.pdmon
   data[i,*] = y[*]
   error[i,*] = dy[*]
   heap_free,d
endfor
wbm = data[17,*]
tbm = data[18,*]
ret = onew->set(data = data,error = error,fc = mon,indep_var = indep_var, $
         wbm = wbm,tbm = tbm)
return,onew
end