; $Id: dm_to_string.pro,v 1.14 2014/12/15 21:56:19 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_to_string
;
; PURPOSE:
;  convert numbers to string
;  if number is !values.f_nan then return ''
;
; CATEGORY:
;  general
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;keywords:
;   addsemicolon: if set, add a semicolun to the output header string only, number variable is a string
;   data:         used for constructing header string, number variable is a string   
;   date:         if set, then number is the unix epoch(since 1970,1,1 00:00:00 utc), return date string in USA local time unless utc or est keyword is set
;   dottop:       if set, replace . with letter p
;   est:          if set, the date is in EST or EDT
;   exponent:     if set, use **e** format
;   format:       used for constructing header string only, number variable is a string
;   int:          if set, the string contains no decimal digits
;   java:         if set, the date string is in java format "yyyy-MM-dd'T'HH:mm:ssZ" if number is integer, "yyyy-MM-dd'T'HH:mm:ss.SSSZ" if number is double
;   negton:       if set, replace - with letter n
;   nozeroend:    if set, no trailing .0 at all
;   overflow:     used for constructing header string only, number variable is a string
;   resolution:   a natural number indicating the number of decimal places for resolution in the string
;   separator:    if present, use separator string to combine the string arrays into one string
;   strlength:    prescribed string length, if the returned string length is less than strlen, blank spaces are filled in
;   utc:          if set, the date is in UTC date
function dm_to_string,number,addsemicolon=addsemicolon,data=data,date=date,dottop=dottop,est=est,exponent=exponent,format=format,int=int,java=java,negton=negton,nozeroend=nozeroend,$
    overflow=overflow,resolution=resolution,separator=separator,strlength=strlength,utc=utc
    tmp = size(number,/type)
    if tmp eq 7 then begin  ;string
       tmpnumber = strtrim(number,2)
       if n_elements(data) eq 0 then return,tmpnumber
       if n_elements(format) eq 0 then format = '(g)'
       if n_elements(overflow) eq 0 then overflow = 0
       header = ''
       for i=0,n_elements(number)-1 do begin
           overflow = strlen(string(data[i<(n_elements(data)-1)],format=format[i<(n_elements(format)-1)]))-strlen(tmpnumber[i])-overflow
           header   = header+strjoin(replicate(' ',1>overflow))+tmpnumber[i]
           overflow = 0>(1-overflow)
       endfor
       if keyword_set(addsemicolon) then begin
          if strmid(header,0,1) eq ' ' then strput,header,';',0 else header=';'+header
       endif
       return,header
    endif
    n_num = n_elements(number)
    n_end = 0   ;for smart '.0' ending
    if n_num eq 0 then return, ''
    if n_num eq 1 then str = '' else str = strarr(size(number,/dimension))
    tmpnumber = number
    if tmp eq 1 then tmpnumber = fix(number)   ;number is byte
    if n_elements(resolution) ne 0 then begin
       myresolution = 0>(fix(resolution[0]))
       if (~keyword_set(int)) and (~keyword_set(exponent)) then begin
          ind = where(finite(tmpnumber),count)
          if count gt 0 then tmpnumber[ind] = round(tmpnumber[ind]*((10.0d)^myresolution)+(0.0001d),/L64)/((10.0d)^myresolution)
       end
    endif else myresolution = 1
    if keyword_set(date) then begin
       cdf_epoch,epoch_0,1970,1,1,0,0,0,/compute_epoch
       tmp_date = systime()
       if abs(dm_to_number(tmp_date,/epoch)-dm_to_number(systime(/utc),/epoch)) le 1 then dst0 = 0 else begin
          tmp   = stregex(tmp_date,'([a-z]{3}) +([0-9]{1,2}) +([0-9]{1,2})\:([0-9]{1,2})\:([0-9]{1,2}) +([0-9]{4})',/fold_case,/subexpr,len=len)
          dow   = (where(['Mon','Tue','Wed','Thu','Fri','Sat','Sun'] eq strmid(tmp_date,0,3)))[0]
          month = (where(['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'] eq strmid(tmp_date,tmp[1],len[1])))[0]+1
          day   = dm_to_number(strmid(tmp_date,tmp[2],len[2]),/int)
          dow   = ((35+dow+1-day) mod 7)
          year  = dm_to_number(strmid(tmp_date,tmp[6],len[6]))
          hour  = dm_to_number(strmid(tmp_date,tmp[3],len[3]))+dm_to_number(strmid(tmp_date,tmp[4],len[4]))/60.+dm_to_number(strmid(tmp_date,tmp[5],len[5]))/3600.
          ;check day light saving time, US east only
          ;dst0  = (year gt 2023) or ((year eq 2023) and (month gt 3)) or ((month gt 3) and (month lt 11)) or ((month eq 3) and ((day+dow gt 14) or ((day+dow eq 14) and (hour ge 2)))) or ((month eq 11) and ((day+dow lt 7) or ((day+dow eq 7) and (hour lt 2))))
          dst0  = ((month gt 3) and (month lt 11)) or ((month eq 3) and ((day+dow gt 14) or ((day+dow eq 14) and (hour ge 2)))) or ((month eq 11) and ((day+dow lt 7) or ((day+dow eq 7) and (hour lt 2))))
       endelse
    endif
    for i=0L,n_num-1 do begin
        if finite(tmpnumber[i]) then begin
           if keyword_set(date) then begin
              if keyword_set(utc) then begin
                 diff = 0
                 tzone = ([' UTC','+00:00'])[keyword_set(java)]
              endif else begin
                 ;figure out the time zone
                 tmp_date = systime(elapsed=fix(tmpnumber[i],type=size(0ll,/type)))
                 if keyword_set(est) then diff = -18000 $
                 else diff = dm_to_number(tmp_date,/epoch)-fix(tmpnumber[i],type=size(0ll,/type))-dst0*3600
                 dow  = (where(['Mon','Tue','Wed','Thu','Fri','Sat','Sun'] eq strmid(tmp_date,0,3)))[0]
                 cdf_epoch,(tmpnumber[i]+diff)*(1000d)+epoch_0,year,month,day,hour,minute,second,milli,/breakdown_epoch
                 hour = hour+minute/60.+(second+milli/1000.)/3600.
                 dow  = ((35+dow+1-day) mod 7)
                 ;dst1 = (year gt 2023) or ((year eq 2023) and (month gt 3)) or ((month gt 3) and (month lt 11)) or ((month eq 3) and ((day+dow gt 14) or ((day+dow eq 14) and (hour ge 2)))) or ((month eq 11) and ((day+dow lt 7) or ((day+dow eq 7) and (hour lt 2))))    
                 dst1 = ((month gt 3) and (month lt 11)) or ((month eq 3) and ((day+dow gt 14) or ((day+dow eq 14) and (hour ge 2)))) or ((month eq 11) and ((day+dow lt 7) or ((day+dow eq 7) and (hour lt 2))))
                 tmp1 = -diff/3600.
                 if ((tmp1 mod 1) eq 0) and (tmp1 ge 4) and (tmp1 le 10) then begin ;all US time zones
                    tmp1 = fix(tmp1)
                    if keyword_set(java) then tzone = '-'+string(tmp1-dst1,format='(i02)')+':00' $
                    else tzone = (['ast','adt','est','edt','cst','cdt','mst','mdt','pst','pdt','akst','akdt','hst','hdt'])[2*(tmp1-4)+dst1]
                 endif else tzone = ''
                 if (strlen(tzone) ne 0) and dst1 then diff = diff+dst1*3600
              endelse
              cdf_epoch,(tmpnumber[i]+diff)*(1000d)+epoch_0,year,month,day,hour,minute,second,milli,/breakdown_epoch
              str[i] = string(year,format='(i04,"-")')+string(month,format='(i02,"-")')+string(day,format='(i02)')+([' ','T'])[keyword_set(java)]+string(hour,format='(i02,":")')+$
                       string(minute,format='(i02,":")')+string(second,format='(i02)')+((milli eq 0)?'':string(milli,format='(".",i03)'))+tzone
           endif else if keyword_set(int) then str[i] = strtrim(string(fix(tmpnumber[i],type=14)),2) $
           else begin
              if keyword_set(exponent) then begin
                 if n_elements(resolution) ne 0 then str[i] = strtrim(string(tmpnumber[i],format='(e'+dm_to_string(10+myresolution)+'.'+dm_to_string(myresolution)+')'),2)  $
                 else str[i] = strtrim(string(tmpnumber[i],format='(e)'),2) 
              endif else begin
                 str[i] = strtrim(string(tmpnumber[i]),2)
              endelse
              ;get rid of leading 0's and '+'
              tmp  = strmid(str[i],0,1)
              head = (tmp eq '-')?'-':''
              if (tmp eq '+') or (tmp eq '-') then str[i] = strmid(str[i],1,strlen(str[i])-1)
              while (strmid(str[i],0,1) eq '0') and (strlen(str[i]) ne 1)  do str[i] = strmid(str[i],1,strlen(str[i])-1)
              if strmid(str[i],0,1) eq '.' then str[i] = '0'+str[i]
              str[i] = head+str[i]
              ;get rid of trailing 0's for floats
              if strpos(str[i],'e') ne -1 then begin
                 tmp  = strsplit(str[i],'e',/extract,/preserve)
                 tmp1 = dm_to_string(fix(tmp[0],type=5))
                 while (strmid(tmp1,1,2,/reverse) eq '10') and ~stregex(tmp1,'\.',/boolean) do begin ;'10' or '-10'
                    tmp1 = strmid(tmp1,0,strlen(tmp1)-1)
                    tmp[1] = tmp[1]+1
                 endwhile
                 str[i] = tmp1+'e'+dm_to_string(fix(tmp[1],type=2))
              endif else begin
                 tmp1 = strpos(str[i],'.')
                 if tmp1 ne -1 then str[i] = str[i]+' ' ;add a trailing empty space
                 tmp = strsplit(str[i],'[0]+ ',/reg,/extract)   ;one or more 0 and one blank space
                 str[i] = strtrim(tmp[0],2)
                 if strmid(str[i],0,1,/reverse_offset) eq '.' then begin
                    if keyword_set(nozeroend) then str[i] = strmid(str[i],0,strlen(str[i])-1)
                    n_end = n_end+1
                 endif
                 if (tmp1 ne -1) and ~keyword_set(nozeroend) then begin
                    tmp2 = strlen(str[i])
                    if (n_num gt 1) or (tmp1 eq tmp2-1) then $ 
                       for j=0,myresolution-(tmp2-tmp1-1)-1 do str[i] = str[i]+'0'
                 endif 
              endelse
           endelse
        endif else n_end = n_end+1
    endfor
    if (n_end eq n_num) and ~keyword_set(nozeroend) then begin
       for i=0L,n_num-1 do str[i] = strmid(str[i],0,strlen(str[i])-myresolution-1)  ;get rid of '.0' ending
    endif
    if n_elements(strlength) eq 1 then begin
       if strlength lt 0 then strlength = 0
       for i=0L,n_num-1 do begin
           tmp2 = strlen(str[i])
           for j=0,strlength-tmp2-1 do str[i] = str[i]+' '
       endfor
    endif
    if n_num eq 1 then str=str[0] else begin
       if n_elements(separator) eq 1 then begin
          tmpstr = str[0]
          for i=1L,n_num-1 do begin
              tmp = stregex(tmpstr,' *$') ;separator will be inserted before the trailing blanks
              tmpstr = strmid(tmpstr,0,tmp)+separator+strmid(tmpstr,tmp,strlen(tmpstr)-tmp)+str[i]
          endfor
          str = tmpstr
       endif 
    endelse
    if keyword_set(dottop) then begin
       for i=0,n_elements(str)-1 do begin
           if stregex(str[i],'\.',/boolean) then str[i] = strjoin(strsplit(str[i],'.',/extract,/preserve_null),'p')
       endfor
    endif
    if keyword_set(negton) then begin
       for i=0,n_elements(str)-1 do begin
           if stregex(str[i],'\-',/boolean) then str[i] = strjoin(strsplit(str[i],'-',/extract,/preserve_null),'n')
       endfor
    endif
    return,str
end