; $Id: dm_macs_spurion.pro,v 1.4 2016/03/07 21:04:01 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_he_recoil
;
; PURPOSE:
;  plot MACS spurions
;  1. caused by gas recoil scattering assuming fixed volume of gas, loaded at 1 atm at 300 K
;     gas recoil scattering can be described by G.L. Squires Book, Eq 4.75
;       S(Q,E)=(beta/(4*pi*Er))^(1/2)*exp(-(beta)/(4*pi*Er)*(E-Er)^2)
;     where Er=hbar^2*Q^2/(2*M)

; CATEGORY:
;  dcs_mslice
;
; HISTORY:
;  1/2016: introduced
;  2/2016: Add he3, air and argon for gas recoil calculation
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  March, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;He4 vapor pressure data from https://userweb.jlab.org/~ckeith/TOOLS/HeVaporTable.html, in units of torr
;He3 vapor pressure from Cryogenics 46 (2006) 833–839, in units of torr
;air vapor pressure use Antonine equation http://highered.mheducation.com/sites/dl/free/0072849606/315014/physical_properties_table.pdf
;Ar  vapor pressure from http://www.nist.gov/data/PDFfiles/jpcrd363.pdf
function dm_hevapp,T,Tmax=Tmax,gastype=gastype
    if n_elements(gastype) eq 0 then gastype = 0
    case gastype of
         0: begin   ; 4He
            Ts = [0.9846,1.0582,1.1063,1.1425,1.1721,1.1973,1.2193,1.239,1.2568,1.2731,1.3898,1.4667,1.5257,1.5743,1.616,1.6527,1.6857,1.7158,1.7435,1.7933,1.8373,1.877,1.9132,1.9468,1.9781,2.0075,2.0352,2.0617,2.0869,$
                  2.1111,2.1344,2.1569,2.1786,2.1998,2.2203,2.2403,2.2599,2.279,2.2977,2.3387,2.3793,2.4175,2.4538,2.4884,2.5214,2.553,2.5833,2.6125,2.6407,2.6943,2.7447,2.7922,2.8373,2.8803,2.9213,2.9607,2.9984,3.0348,$
                  3.0699,3.1039,3.1367,3.1686,3.1995,3.2295,3.2588,3.2873,3.3151,3.3422,3.3687,3.3946,3.42,3.4448,3.4691,3.493,3.5164,3.5393,3.5619,3.584,3.6058,3.6272,3.6482,3.6689,3.6893,3.7094,3.7292,3.7487,3.7679,$
                  3.7868,3.8055,3.8239,3.8421,3.8601,3.8778,3.8954,3.9127,3.9297,3.9466,3.9633,3.9798,3.9962,4.0123,4.0283,4.0441,4.0597,4.0752,4.0905,4.1056,4.1206,4.1355,4.1502,4.1648,4.1792,4.1935,4.2077,4.2218]
            Ps = [0.1*(indgen(9)+1),findgen(9)+1.0,10.0+2.0*findgen(20),50.0+5.0*findgen(10),100.0+10.0*findgen(67)]
            if arg_present(Tmax) then Tmax = 4.2218
            return,interpol(Ps,Ts,T)
            end
         1: begin   ; 3He
            if T lt 2.7 then begin
               as = [-2.51017d,9.69821d,-0.286067d,0.20103d,-0.0519116d,0.00532652d]
               b  = 2.248228d
               p  = (exp(total(as*T^[indgen(n_elements(as))-1.0])+b*alog(T)))/133.3
            endif else begin
               tr = T/3.3157
               A  = -3.32211d
               cs = [3.399174d,0,1.924479d,-14.689902d,70.478942d,-118.19003d]
               p  = (1.-A*(1-tr)^1.895-total(cs*(1-tr)^[indgen(n_elements(cs))+1.0]))*899.744
            endelse
            if arg_present(Tmax) then Tmax = 3.3157
            return,p
            end
         2: begin   ; air
            f_o2 = 0.209476  ;mole fraction of O2
            f_n2 = 0.78084   ;mole fraction of N2
            a_n2 = 6.49457 & b_n2 = 255.68  & c_n2 = 266.55
            a_o2 = 6.69144 & b_o2 = 319.013 & c_o2 = 266.697
            if arg_present(Tmax) then Tmax = 90.19
            if T lt 20 then return, 0.0 $
            else return,f_n2*((10.0)^(a_n2-(b_n2/(T-273.15+c_n2))))+f_o2*((10.0)^(a_o2-(b_o2/(T-273.15+c_o2))))
            end
         3: begin   ; Ar
            Tc = 150.6633 & Pc = 4.86e6/133.3
            Ns = [-5.904188529d,1.125495907d,-0.7632579126d,-1.697334376d]
            tau = 1.0-T/Tc
            if arg_present(Tmax) then Tmax = 87.3
            return,Pc*exp(Tc/T*(Ns[0]*tau+Ns[1]*tau^1.5+Ns[2]*tau^3+Ns[3]*tau^6))
            end
         else:begin
            if arg_present(Tmax) then Tmax = 0.0
            return, 0.0
            end
    endcase
end

;S(Q,E)=(beta/(4*pi*Er))^(1/2)*exp(-(beta)/(4*pi*Er)*(E-Er)^2)
;where Er=hbar^2*Q^2/(2*M)
function dm_herecsqe,T,Q,En,gastype=gastype
    case gastype of
         0: hbsqo2m = 0.5218360      ;(81.8042)/(2*!pi)^2*1.008/4.0026  ;hbar^2/(2*M_He4)
         1: hbsqo2m = 0.6925400      ;(81.8042)/(2*!pi)^2*1.008/3.016   ;hbar^2/(2*M_He3)
         2: hbsqo2m = 0.0720988      ;(81.8042)/(2*!pi)^2*1.008/28.97   ;hbar^2/(2*M_Air)
         3: hbsqo2m = 0.0522855      ;(81.8042)/(2*!pi)^2*1.008/39.948  ;hbar^2/(2*M_Ar)
         else:
    endcase
    kb  = 8.617342e-2                                     ;Boltzmann constant in mev.k^-1
    Er  = hbsqo2m*Q^2
    tmp = 1./(kb*T*4.0*!pi*Er)
    return,sqrt(tmp)*exp(-tmp*(En-Er)^2.0)
end

;keywords:
;   gasrecoil:  if set, calculate the gas recoil scattering, the only one implemented 
pro dcs_mslice::dm_macs_spurion,gasrecoil=gasrecoil
    ;catch and ignore all errors in this program, necessary because dm_getkidneyrange() may generate out-of-range error
    catch, myerror
    if myerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg,/error,dialog_parent=self.tlb,/center)
       if obj_valid(mesg) then obj_destroy,mesg
       return
    endif

    if ~keyword_set(gasrecoil) then return
    EiEf = !values.f_nan & T1 = !values.f_nan & T2 = !values.f_nan
    if (finite(self.eief)) then begin
       if self.eief gt 0 then EiEf=dm_to_number(dm_to_string(self.eief,resolution=3))
    endif
    if ptr_valid(self.dataStrPtr) then begin
       if ptr_valid(self.projStrPtr) then tmp = (*self.projStrPtr).temperature $
       else tmp = (*self.dataStrPtr).temperature
       tmp_max = max(tmp[*,self.macstemp_typ],min=tmp_min)
       if tmp_max gt 0 then T2 = tmp_max
       if tmp_min gt 0 then T1 = tmp_min
       tmp = 0
    endif

    fts   = ['from','to','step']
    name  = ['gas type',['fixed '+(['Ei','Ef'])[self.instrgeom],'E '+fts]+' (meV)']
    ans   = [0,dm_to_number(dm_to_string([EiEf,self.estartend],resolution=3))]
    ok    = dm_getkidneyrange(5,eimat=eimat)
    info  = ['Ei'+(['','=Ef+E'])[self.instrgeom]+' should be in the range of ['+dm_to_string(min(eimat))+', '+dm_to_string(max(eimat))+'] meV.','T2 is optional.',$
             (['Leave A3 fields empty for powder projection.','Switch the sample type to single crystal for single crsystal projection.'])[self.samp_typ eq 0]]
    if self.samp_typ gt 0 then begin
       self->dm_check_crystparm,U1=U1,U2=U2,U3=U3,error=error
       if keyword_set(error) then return
       name = [name,'A3 '+fts]
       ans  = [ans,dm_to_number(dm_to_string(self.psi[0:2],resolution=3))]
    endif
    name  = [name,'T1 (K)','T2 (K)','S(Q,omega) type']
    ans   = [ans,dm_to_number(dm_to_string([T1,T2],resolution=3)),0.0]
    nans  = n_elements(ans)
    notok = 1b
    is_droplist = bytarr(n_elements(name))+1b & is_droplist[1:nans-2] = 0 
    while notok do begin
       ans = dm_dialog_input(name+':',default=ans,dialog_parent=self.tlb,/align_center,info=info,cancel=cancel,droplist_content=[ptr_new(['helium-4','helium-3','air','argon']),ptr_new(['T1-T2','S(Q,omega,T)'])],$
             is_droplist=is_droplist,xsize=90,/return_number)
       if keyword_set(cancel) then return
       if (total(finite(ans,/nan)) eq total(finite(ans[nans-2],/nan))) or ((self.samp_typ gt 0) and (total(finite(ans,/nan)) eq total(finite(ans[[5,6,7,nans-2]],/nan)))) then $
          notok = (ans[1] le 0) or ((ans[nans-1] eq 0) and (finite(ans[nans-2]) and (ans[nans-3] eq ans[nans-2])))
    endwhile
    samp_typ = self.samp_typ
    
    Prt   = 1.0       ;pressure of the He gas of the sample cell at room temperature of 300K, in units of atm, no user input
    gas   = ans[0]
    EiEf  = ans[1]
    Eran  = [min(ans[2:3]),max(ans[2:3])]
    Estep = abs(ans[4])
    T1    = ans[nans-3]
    T2    = ans[nans-2]
    sqet  = (round(ans[nans-1]) and finite(T2))
    if Estep eq 0 then Eran = [Eran[0],Eran[0]]
    if samp_typ gt 0 then begin  ;single crystal type
       if total(finite(ans[5:7],/nan)) eq 0 then begin
          a3ran  = [min(ans[5:6]),max(ans[5:6])]
          a3step = abs(ans[7])
          if a3step eq 0 then a3ran = [a3ran[0],a3ran[0]]
       endif else samp_typ = 0
    endif
    if keyword_set(sqet) then begin
       nT = 40<(ceil(abs(T2-T1))+1)
       if nT eq 1 then Ts = T1 else Ts = T1+findgen(nT)/(nT-1.0)*(T2-T1)
    endif
    instrument = (['Helium'+['','-3']+' Recoil','Air Scattering','Argon Recoil'])[gas]
    
    tmp   = dm_hevapp(2,Tmax=Tmax,gastype=gas)
    title = (['Ei','Ef'])[self.instrgeom]+'='+dm_to_string(EiEf)+' meV'
    n_E   = (Estep eq 0)?0:(floor((Eran[1]-Eran[0])/Estep))
    Es    = Eran[0]+findgen(n_E+1)*Estep
    if self.instrgeom eq 1 then begin ;inverse geometry
       Ef = replicate(EiEf,n_elements(Es))
       Ei = Es+EF
    endif else begin
       Ei = replicate(EiEf,n_elements(Es))
       Ef = Ei-Es
    endelse
    
    mesg = obj_new('dm_progress',title='Please wait',message='Calculating '+(['helium','helium-3','air','argon'])[gas]+' recoil scattering...',group_leader=self.tlb) 
    
    if samp_typ ne 0 then begin ;single crystal projection
       det_tt   = findgen(20)*8.0 
       det_psi  = fltarr(20)
       ulabel   = (['H','K','L'])[(where(self.view_u1 ne 0))[0]]
       ulabel   = dm_set_viewlabel(ulabel,self.view_u1)
       vlabel   = (['H','K','L'])[(where(self.view_u2 ne 0))[0]]
       vlabel   = dm_set_viewlabel(vlabel,self.view_u2)
       label    = ['Intensity','dI',ulabel,vlabel,'E']
       unit     = ['arb. unit','','','','meV']
       is_uniq  = [0,0,0,0,1]
       n_a3     = (a3step eq 0)?0:(floor((a3ran[1]-a3ran[0])/a3step))
       a3       = 90.0-(a3ran[0]+findgen(n_a3+1)*a3step+(finite(self.orie_psi,/nan)?0.0:(self.orie_psi)))
       sqe      = 0.0 & qx1 = 0.0 & qy1 = 0.0 & es1 = 0.0 & ts1 = 0.0 & sqe2 = 0.0
       if keyword_set(sqet) then begin
          for i=0,n_elements(Es)-1 do begin              
              kidran  = dm_getkidneyrange(Ei[i])
              kstep   = dm_optimizekidstep(0.8,kidran,eq_step=eq_step)
              n_kid   = (kstep eq 0)?0:(floor(abs(kidran[1]-kidran[0])/kstep))
              n_dat   = (n_kid+1)*(n_a3+1)
              kidney  = kidran[0]+findgen(n_kid+1)*kstep
              kidney  = reform(kidney#(fltarr(n_a3+1)+1),n_dat)
              tmpa3   = reform((fltarr(n_kid+1)+1)#a3,n_dat)
              tmpEi   = replicate(Ei[i],n_dat)
              tmpEf   = replicate(Ef[i],n_dat)
              Q0      = dm_proj_spec_samp(tmpEi,tmpEf,det_tt,det_psi,type=0,kidney=kidney)                      ;Q0[ndat,ndet]
              Q1      = dm_proj_spec_samp(tmpEi,tmpEf,det_tt,det_psi,psi=tmpa3,type=1,is_spe=0,kidney=kidney)   ;Q1[3,ndat,ndet] in sample frame
              QV      = dm_proj_view(Q1,U1,U2,U3)
              for j=0,n_elements(Ts)-1 do begin
                  if Ts[j] ge Tmax then n1 = 1.0 else n1 = 1.0<(dm_hevapp(Ts[j],gastype=gas)*300./(Prt*760.*Ts[j])) ;n1 can never exceed 1.0
                  sqe = [sqe,n1*reform(dm_herecsqe(Ts[j],Q0,Es[i],gastype=gas),n_dat*20)]
                  qx1 = [qx1,reform(QV[0,*,*],n_dat*20)]
                  qy1 = [qy1,reform(QV[1,*,*],n_dat*20)]
                  es1 = [es1,replicate(Es[i],n_dat*20)]
                  ts1 = [ts1,replicate(Ts[j],n_dat*20)]
              endfor
              mesg->update,message='Calculating '+(['helium','helium-3','air','argon'])[gas]+' recoil scattering...'+dm_to_string(round(float(i+1)/n_elements(Ei)*100))+'%'
              if (~ obj_valid(mesg)) then return
          endfor
          label = [label,'T']
          unit  = [unit,'K']
          data  = fltarr(n_elements(sqe)-1,6)
          data[*,5] = ts1[1:*]
          is_uniq = [is_uniq,1]
       endif else begin    
          if T1 ge Tmax then n1 = 1.0 else n1 = 1.0<(dm_hevapp(T1,gastype=gas)*300./(Prt*760.*T1))       ;n1 can never exceed 1.0
          if finite(T2) then begin
             if T2 ge Tmax then n2 = 1.0 else n2 = 1.0<(dm_hevapp(T2,gastype=gas)*300./(Prt*760.*T2))    ;n2 can never exceed 1.0
          endif
          for i=0,n_elements(Ei)-1 do begin
              kidran  = dm_getkidneyrange(Ei[i])
              kstep   = dm_optimizekidstep(0.8,kidran,eq_step=eq_step)
              n_kid   = (kstep eq 0)?0:(floor(abs(kidran[1]-kidran[0])/kstep))
              n_dat   = (n_kid+1)*(n_a3+1)
              kidney  = kidran[0]+findgen(n_kid+1)*kstep
              kidney  = reform(kidney#(fltarr(n_a3+1)+1),n_dat)
              tmpa3   = reform((fltarr(n_kid+1)+1)#a3,n_dat)
              tmpEi   = replicate(Ei[i],n_dat)
              tmpEf   = replicate(Ef[i],n_dat)
              Q0      = dm_proj_spec_samp(tmpEi,tmpEf,det_tt,det_psi,type=0,kidney=kidney)                      ;Q0[ndat,ndet]
              Q1      = dm_proj_spec_samp(tmpEi,tmpEf,det_tt,det_psi,psi=tmpa3,type=1,is_spe=0,kidney=kidney)   ;Q1[3,ndat,ndet] in sample frame
              QV      = dm_proj_view(Q1,U1,U2,U3)                                                               ;project onto viewing axis
              sqe     = [sqe,reform(dm_herecsqe(T1,Q0,Es[i],gastype=gas),n_dat*20)]
              qx1     = [qx1,reform(QV[0,*,*],n_dat*20)]
              qy1     = [qy1,reform(QV[1,*,*],n_dat*20)]
              es1     = [es1,replicate(Es[i],n_dat*20)]
              if finite(T2) then sqe2 = [sqe2,reform(dm_herecsqe(T2,Q0,Es[i],gastype=gas),n_dat*20)]
              mesg->update,message='Calculating '+(['helium','helium-3','air','argon'])[gas]+' recoil scattering...'+dm_to_string(round(float(i+1)/n_elements(Ei)*100))+'%'
              if (~ obj_valid(mesg)) then return
          endfor
          if finite(T2) then sqe = n1*temporary(sqe)-n2*temporary(sqe2)
          data = fltarr(n_elements(sqe)-1,5)
       endelse 
       data[*,0] = sqe[1:*]
       data[*,2] = qx1[1:*]
       data[*,3] = qy1[1:*]
       data[*,4] = es1[1:*] 
    endif else begin      ;powder projection
       label   = ['Intensity','dI','|Q|','E']
       unit    = ['arb. unit','','\AA!U-1!N','meV']
       is_uniq = [0,0,0,1]
       sqe     = 0.0 & qs1 = 0.0 & es1 = 0.0 & ts1 = 0.0 & sqe2 = 0.0
       if keyword_set(sqet) then begin
          for i=0,n_elements(Ei)-1 do begin
              kidran  = dm_getkidneyrange(Ei[i])
              tthran  = [kidran[0]-76.,kidran[1]-76.+152.]
              tth     = tthran[0]+findgen(201)/200.*(tthran[1]-tthran[0])
              Qs      = dcs_cryst_align_qlength2(Ei[i],tth, Ef=Ef[i])
              for j=0,n_elements(Ts)-1 do begin
                  if Ts[j] ge Tmax then n1 = 1.0 else n1 = 1.0<(dm_hevapp(Ts[j],gastype=gas)*300./(Prt*760.*Ts[j])) ;n1 can never exceed 1.0
                  sqe = [sqe,n1*dm_herecsqe(Ts[j],Qs,Es[i],gastype=gas)]
                  qs1 = [qs1,Qs]
                  es1 = [es1,replicate(Es[i],n_elements(Qs))]
                  ts1 = [ts1,replicate(Ts[j],n_elements(Qs))]
              endfor
              mesg->update,message='Calculating '+(['helium','helium-3','air','argon'])[gas]+' recoil scattering...'+dm_to_string(round(float(i+1)/n_elements(Ei)*100))+'%'
              if (~ obj_valid(mesg)) then return
          endfor
          label = [label,'T']
          unit  = [unit,'K']
          data  = fltarr(n_elements(sqe)-1,5)
          data[*,4] = ts1[1:*]
          is_uniq = [is_uniq,1]
       endif else begin
          if T1 ge Tmax then n1 = 1.0 else n1 = 1.0<(dm_hevapp(T1,gastype=gas)*300./(Prt*760.*T1))       ;n1 can never exceed 1.0 
          if finite(T2) then begin
             if T2 ge Tmax then n2 = 1.0 else n2 = 1.0<(dm_hevapp(T2,gastype=gas)*300./(Prt*760.*T2))    ;n2 can never exceed 1.0
          endif
          for i=0,n_elements(Ei)-1 do begin
              kidran  = dm_getkidneyrange(Ei[i])
              tthran  = [kidran[0]-76.,kidran[1]-76.+152.]
              tth     = tthran[0]+findgen(201)/200.*(tthran[1]-tthran[0])
              Qs      = dcs_cryst_align_qlength2(Ei[i],tth, Ef=Ef[i])
              sqe     = [sqe,dm_herecsqe(T1,Qs,Es[i],gastype=gas)]
              qs1     = [qs1,Qs]
              es1     = [es1,replicate(Es[i],n_elements(Qs))]
              if finite(T2) then sqe2 = [sqe2,dm_herecsqe(T2,Qs,Es[i],gastype=gas)]
          endfor
          if finite(T2) then sqe = n1*temporary(sqe)-n2*temporary(sqe2)
          data = fltarr(n_elements(sqe)-1,4)
       endelse
       data[*,0] = sqe[1:*]
       data[*,2] = qs1[1:*]
       data[*,3] = es1[1:*]
    endelse
    
    if ~keyword_set(sqet) then begin
       if finite(T2,/nan) then title = title+', T='+dm_to_string(T1)+' K' else title = title+', '+dm_to_string(T1)+' K-'+dm_to_string(T2)+' K'
    endif
    
    if obj_valid(mesg) then obj_destroy,mesg
    
    ;find existing mslice data viewer and use it
    all = obj_valid(count=count)
    for i=0,count-1 do begin
        if isa(all[i],'dcs_mslice') then begin
           all[i]->getproperty,ftype=ftype,uname=uname
           if (ftype eq 'mslicedata') and (uname eq 'macs spurion') then begin
              all[i]->reset_data,data=data,label=label,instrument=instrument,is_uniq=is_uniq,unit=unit,title=title,/no_copy
              return
           endif
        endif
    endfor
    dataPtr = ptr_new({data:data,label:label,unit:unit,is_uniq:is_uniq,title:title,instrument:instrument})
    obj_mslice = obj_new('dcs_mslice',dataPtr=dataPtr,uname='macs spurion',group_leader=self.tlb)
    obj_mslice->setproperty,tickdir=self.tdir_choice
end