; $Id:$
;#######################################################################
;
; NAME:
;  dcs_whitebeam
;
; PURPOSE:
;  this program calculates the transmission from two set of DCS white beam files
;
; CATEGORY:
;  dcs tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  January, 2024
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;clear all objects and widget hierarchy
pro dcs_whitebeam_Exit,tlb
    widget_control,tlb,get_uvalue=state,/no_copy
    obj_destroy,[state.plot1,state.plot2]
    ptr_free,state.dataPtr,state.resultPtr
    widget_control,tlb,/destroy
end

pro dcs_whitebeam_loadfile,state,empty=empty,browse=browse
    ind = keyword_set(empty)
    if keyword_set(browse) then begin
       path = state.dirs[0]
       file = dm_choose_file('',dialog_parent=state.tlb,path=path,/read,/multiple_files)
       for i=1,n_elements(file)-1 do file[0] = file[0]+'&&'+file[i]
       if strlen(file[0]) ne 0 then begin
          if state.files[ind] eq file[0] then return
          state.files[ind] = file[0]
          state.dirs[0]    = path
          widget_control,state.fileText[ind],set_value=file[0]
          load = 1b
       endif
    endif else begin
       widget_control,state.fileText[ind],get_value=tmp
       tmp = strtrim(tmp,2)
       if state.files[ind] eq tmp then return
       state.files[ind] = tmp
       tmp = strtrim((strsplit(tmp,'&&',/extract,/regex))[0],2)
       if (file_test(tmp,/read) and ~file_test(tmp,/directory)) or (tmp eq '') then begin
          load = 1b
          widget_control,state.fileText[1-ind],set_text_select=0
       endif
    endelse
 
    if ~keyword_set(load) then return

    if stregex(state.files[ind],'&&',/boolean) then files = strsplit(state.files[ind],'&&',/extract,/regex) else files = state.files[ind]
    files = strtrim(files,2)
    ptr_free,state.dataPtr[ind]
    widget_control,([state.svsmBut,state.svemBut])[ind],sensitive=0
    for i=0,3 do begin
        state.pgind[i] = -1
        widget_control,state.chans[i],set_value=''
    endfor
    if total(strlen(files)) ne 0 then begin
       davePtr = dm_define_pointer() & omithisto=1 & printinfo=0 & filetype='binary'
       bm1 = 0.0 & bm3 = 0.0 & dur = 0.0
       for i=0,n_elements(files)-1 do begin
           if strlen(files[i]) eq 0 then continue
           dcs_readdatafile,files[i],omithisto,printinfo,filetype,davePtr=davePtr
           this = dm_locate_datastrptr(davePtr)
           bm1  = bm1+total((*(*this).specificPtr).histohigh[0:999,3])
           bm3  = bm3+(*(*this).specificPtr).histohigh[0:999,5]
           dur  = dur+(*(*this).specificPtr).duration
           if n_elements(comment) eq 0 then comment = (*(*this).specificPtr).comments
       endfor
       heap_free,davePtr
       state.dataPtr[ind] = ptr_new({bm1:bm1,bm3:bm3,duration:dur,comment:comment})
       widget_control,([state.svsmBut,state.svemBut])[ind],/sensitive
    endif
    dcs_whitebeam_plot,state
end

pro dcs_whitebeam_plot,state
    ptr_free,state.resultPtr
    state.plot1->erase
    state.plot2->erase
    spl1menu = widget_info(state.tlb,find_by_uname='spl1menu')
    spl2menu = widget_info(state.tlb,find_by_uname='spl2menu')
    widget_control,spl1menu,sensitive=0
    widget_control,spl2menu,sensitive=0
    if total(ptr_valid(state.dataPtr)) ne 2 then return
    
    if state.bm3 eq 0 then bm3 = (*state.dataPtr[0]).bm3+(*state.dataPtr[1]).bm3 $
    else bm3 = (*state.dataPtr[state.bm3-1]).bm3
    
    if total(state.pgind[0:1] ge 0) ne 2 then begin
       bmd = deriv(indgen(1000),bm3)
       ind1 = (where(bm3 gt max(bm3)/2))[0]
       tmp = min(bmd[ind1:*],ind2)
       tmp = min(bm3[ind1+ind2:ind1+ind2+30],ind3)
       state.pgind[0] = ind1+ind2+ind3
       tmp = min(bmd[state.pgind[0]+40:*],ind1)
       tmp = min(bm3[state.pgind[0]+40+ind1:(state.pgind[0]+70+ind1)<999],ind2)
       state.pgind[1] = state.pgind[0]+40+ind1+ind2
       tmp = linfit(indgen(31),bm3[0:30],measure_errors=sqrt(1>(bm3[0:30])))
       state.pgind[2] = (where(abs(bm3-tmp[0]-indgen(1000)*tmp[1]) gt 20*stddev(bm3[0:30])))[0]
       tmp = where(abs(bm3-tmp[0]-shift(indgen(1000)-969,31)*tmp[1]) gt 20*stddev(bm3[0:30]))
       state.pgind[3] = tmp[n_elements(tmp)-1]
       if state.pgind[2] eq 0   then state.pgind[2] = -1
       if state.pgind[3] eq 999 then state.pgind[3] = -1
       for i=0,3 do widget_control,state.chans[i],set_value=((state.pgind[i] ge 0)?dm_to_string(state.pgind[i]):'')
    endif
    
    state.plot1->setproperty,xdat=indgen(1000),ydat=bm3,xtit='Time Channel',ytit='BM3',psym='circle',/nodraw,cornertxt=(['sample+empty','sample','empty'])[state.bm3]
    for i=0,3 do begin
      if state.pgind[i] ge 0 then state.plot1->add_plot,[state.pgind[i],state.pgind[i]],[-1e20,1e20],color=(['red','magenta','green','green'])[i],linestyle='dashed',psym='no symbol',thick=2,$
        legend=(['PG(004)=','PG(002)=','cutoff!dlower!n=','cutoff!dupper!n=']+dm_to_string(state.pgind))[i]
    endfor
    state.plot1->draw
    widget_control,spl1menu,/sensitive
    if state.norm then factor = (*state.dataPtr[1]).duration/(*state.dataPtr[0]).duration $
    else factor = (*state.dataPtr[1]).bm1/(*state.dataPtr[0]).bm1
    trans = (*state.dataPtr[0]).bm3/(*state.dataPtr[1]).bm3*factor
    dtran = trans*sqrt(1.0/(1>(*state.dataPtr[0]).bm3)+1.0/(1>(*state.dataPtr[1]).bm3))
    wl4   = 3.354 & wl2 = 2*wl4 & chn2 = state.pgind[1] & chn4 = state.pgind[0]
    A     = (wl4*chn2-wl2*chn4)/(chn2-chn4)
    B     = (wl4-A)/chn4
    chn   = findgen(1000)
    ch0   = -A/B
    ;make sure wls are always gt 0
    if B gt 0 then begin
       ind = where(chn le ch0,count)
       if count ne 0 then chn[ind] = chn[ind]+1000.0
    endif else begin
       ind = where(chn ge ch0,count)
       if count ne 0 then chn[ind] = chn[ind]-1000.0
    endelse
    ind   = sort(chn)
    chn   = chn[ind]
    trans = trans[ind]
    dtran = dtran[ind]
    if (state.pgind[2] ge 0) and (state.pgind[2] le 999) then begin
       ind   = where(chn ge state.pgind[2])
       chn   = chn[ind]
       trans = trans[ind]
       dtran = dtran[ind]
    endif
    if (state.pgind[3] ge 0) and (state.pgind[3] le 999) then begin
       ind   = where(chn le state.pgind[3])
       chn   = chn[ind]
       trans = trans[ind]
       dtran = dtran[ind]
    endif
    wls   = A+B*chn
    ind   = where(finite(trans),count)
    if count eq 0 then return
    wls   = wls[ind]
    trans = trans[ind]
    dtran = dtran[ind]
    if state.xaxis then wls = 81.8042/(wls)^2    ;wavelength->energy
    state.resultPtr = ptr_new({x:wls,y:trans,dy:dtran})
    if ~state.showebar then tmp = temporary(dtran)
    if (float(!version.release) ge 6.0) then begin
       if stregex(state.files[0],'&&',/boolean) then file_sampl = strsplit(state.files[0],'&&',/extract,/regex) else file_sampl = state.files[0]
       if stregex(state.files[1],'&&',/boolean) then file_nosam = strsplit(state.files[1],'&&',/extract,/regex) else file_nosam = state.files[1]
       tmp = file_basename(file_sampl,strmid(file_sampl[0],strpos(file_sampl[0],'.')))
       if n_elements(file_sampl) gt 1 then filename = tmp[0]+'-'+tmp[n_elements(tmp)-1] else filename = tmp[0]
       cornertxt = 'sample: '+filename+'!c'
       tmp = file_basename(file_nosam,strmid(file_nosam[0],strpos(file_nosam[0],'.')))
       if n_elements(file_nosam) gt 1 then filename = tmp[0]+'-'+tmp[n_elements(tmp)-1] else filename = tmp[0]
       cornertxt = cornertxt+'empty:  '+filename
    endif
    
    state.plot2->setproperty,xdat=wls,ydat=trans,yerr=dtran,psym='circle',linestyle='solid',xtit=(['\lambda (\AA)','E (meV)'])[state.xaxis],ytit='Transmission',$
       title='normalized by '+(['beam monitor','duration'])[state.norm],cornertxt=cornertxt
    widget_control,spl2menu,/sensitive
end

;index = 0-plot1 1-plot2 2-sample(bm1, duration, bm3) 3-empty(bm1, duration, bm3)
pro dcs_whitebeam_saveascii,state,index
    path = state.dirs[1]
    filename = dm_choose_file('txt',/write,dialog_parent=state.tlb,title='Save '+(['BM3','transmission','sample','empty'])[index]+' data as an ASCII file.',path=path)
    if strlen(filename) eq 0 then return
    openw,unit,filename,/get_lun,error=openerr
    if openerr ne 0 then begin
       ok = dialog_message("Can't write in "+filename[0],/error,dialog_parent=state.tlb,/center)
    endif else begin
       widget_control,/hourglass  ;busy signal
       switch index of 
            0: begin
               printf,unit,';    channel   BM3('+(['sample+empty','sample','empty'])[state.bm3]+')'
               if state.bm3 eq 0 then for i=0,999 do printf,unit,i,' ',(*state.dataPtr[0]).bm3[i]+(*state.dataPtr[1]).bm3[i] $
               else for i=0,999 do printf,unit,i,' ',(*state.dataPtr[state.bm3-1]).bm3[i] 
               break
               end
            1: begin
               printf,unit,';      '+(['wavelength','energy'])[state.xaxis]+'   transmission    error'
               for i=0,n_elements((*state.resultPtr).x)-1 do printf,unit,(*state.resultPtr).x[i],' ',(*state.resultPtr).y[i],' ',(*state.resultPtr).dy[i]
               break
               end
            2: 
            3: begin
               printf,unit,';'+(*state.dataPtr[index-2]).comment
               printf,unit,';bm1'
               printf,unit,(*state.dataPtr[index-2]).bm1
               printf,unit,';duration'
               printf,unit,(*state.dataPtr[index-2]).duration
               printf,unit,';bm3'
               for i=0,999 do printf,unit,(*state.dataPtr[index-2]).bm3[i]
               break
               end
            else:
       endswitch
       free_lun,unit
       state.dirs[1] = path
       state.plot1->setproperty,path=path
       state.plot2->setproperty,path=path
    endelse
end

;handle channel textbox events
pro dcs_whitebeam_updatechan,state,ind
    widget_control,state.chans[ind],get_value=tmp0
    tmp = dm_to_number(tmp0)
    if tmp eq state.pgind[ind] then return
    if finite(tmp,/nan) then tmp=-1
    if tmp ge 0 then begin
       state.pgind[ind] = dm_to_number(tmp,/int)
    endif else begin
       state.pgind[ind] = -1
       widget_control,state.chans[ind],set_value=''
    endelse
    dcs_whitebeam_plot,state
    for i=0,3 do widget_control,state.chans[i],set_text_select=0
    widget_control,state.chans[ind],set_text_select=[strlen(tmp0),0],/input_focus
end

;event handler
pro dcs_whitebeam_event,event
    compile_opt IDL2    
    WIDGET_CONTROL,/HOURGLASS
    widget_control,event.handler,get_uvalue=state,/no_copy
 
    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=state.tlb,!error_state.msg,/error,/center)
       catch,/cancel
       widget_control,event.handler,set_uvalue=state,/no_copy
       return
    end
    
    case event.id of
         state.tlb:         widget_control,event.id,scr_xsize=state.geom[0],scr_ysize=state.geom[1]
         state.doneBut:     begin
                            widget_control,event.handler,set_uvalue=state,/no_copy
                            widget_control,event.handler,/destroy
                            return
                            end
         state.save1But[0]: state.plot1->saveas,'ps'
         state.save1But[1]: state.plot1->saveas,'pdf'
         state.save1But[2]: state.plot1->saveas,'jpeg'
         state.save1But[3]: state.plot1->saveas,'gif'
         state.save1But[4]: state.plot1->saveas,'tiff'
         state.save1But[5]: state.plot1->saveas,'bmp'
         state.save1But[6]: state.plot1->saveas,'png'
         state.save1But[7]: state.plot1->saveas,'ppm'
         state.save1But[8]: dcs_whitebeam_saveascii,state,0
         state.save2But[0]: state.plot2->saveas,'ps'
         state.save2But[1]: state.plot2->saveas,'pdf'
         state.save2But[2]: state.plot2->saveas,'jpeg'
         state.save2But[3]: state.plot2->saveas,'gif'
         state.save2But[4]: state.plot2->saveas,'tiff'
         state.save2But[5]: state.plot2->saveas,'bmp'
         state.save2But[6]: state.plot2->saveas,'png'
         state.save2But[7]: state.plot2->saveas,'ppm'
         state.save2But[8]: dcs_whitebeam_saveascii,state,1     
         state.svsmBut:     dcs_whitebeam_saveascii,state,2
         state.svemBut:     dcs_whitebeam_saveascii,state,3
         state.bmonBut:     if state.norm eq 1 then begin
                            state.norm = 0
                            dm_toggle_menubut,check=event.id,uncheck=state.duraBut
                            dcs_whitebeam_plot,state
                            endif
         state.duraBut:     if state.norm eq 0 then begin
                            state.norm = 1
                            dm_toggle_menubut,check=event.id,uncheck=state.bmonBut
                            dcs_whitebeam_plot,state
                            endif
         state.xwavBut:     if state.xaxis eq 1 then begin
                            state.xaxis = 0
                            dm_toggle_menubut,check=event.id,uncheck=state.xengBut
                            dcs_whitebeam_plot,state
                            endif    
         state.xengBut:     if state.xaxis eq 0 then begin
                            state.xaxis = 1
                            dm_toggle_menubut,check=event.id,uncheck=state.xwavBut
                            dcs_whitebeam_plot,state
                            endif
         state.bm3But[0]:   if state.bm3 ne 0 then begin
                            state.bm3 = 0
                            state.pgind[0] = -1
                            dm_toggle_menubut,check=event.id,uncheck=state.bm3But[[1,2]]
                            dcs_whitebeam_plot,state
                            endif
         state.bm3But[1]:   if state.bm3 ne 1 then begin
                            state.bm3 = 1
                            state.pgind[0] = -1
                            dm_toggle_menubut,check=event.id,uncheck=state.bm3But[[0,2]]
                            dcs_whitebeam_plot,state
                            endif 
         state.bm3But[2]:   if state.bm3 ne 2 then begin
                            state.bm3 = 2
                            state.pgind[0] = -1
                            dm_toggle_menubut,check=event.id,uncheck=state.bm3But[[0,1]]
                            dcs_whitebeam_plot,state
                            endif        
         state.ebarBut:     begin
                            state.showebar = ~state.showebar
                            dm_set_button,event.id,state.showebar
                            dcs_whitebeam_plot,state
                            end
         state.helpBut:     begin
                            info = ['Click the Folder image to browse for files. Multiple files are allowed and will be separated by && in the text box.','','After both sets of files are specified, '+$
                                    'the PG002 and PG004 channel information will be shown in the upper plot, along with lower and upper cutoff channel numbers beyond which the data will be truncated '+$
                                    'since the intensity is too low for those channels. Typical channel numbers for PG004 and PG002 are 225 and 381 respectively. Delete the cutoff channel numbers to '+$
                                    'include the truncated channels. Removing either the PG004 or the PG002 channel will allow the program to recalculate the channel numbers.','','The transimission is '+$
                                    'calcuated as the ratio of BM3 intensities of the two sets of files. The BM3 intensity can be normalized by either beam monitor counts or duration, which can be set '+$
                                    'in the "Options->Normalize To" menu. The result is shown in the lower plot. The transmission can be plotted vs either wavelength or energy, which can be set in the '+$
                                    '"Options->Plot Transmission Vs" menu.','','Plot menu can be accessed by right click over the upper left corner of each plot for more options.']
                            ok = dialog_message(info,/center,/info,dialog_parent=event.handler,title='About DCS White Beam Transmission')
                            end
         state.browBut[0]:  dcs_whitebeam_loadfile,state,/browse                 
         state.browBut[1]:  dcs_whitebeam_loadfile,state,/browse,/empty                
         state.fileText[0]: if event.type le 2 then dcs_whitebeam_loadfile,state
         state.fileText[1]: if event.type le 2 then dcs_whitebeam_loadfile,state,/empty
         state.chans[0]:    if event.type le 2 then dcs_whitebeam_updatechan,state,0
         state.chans[1]:    if event.type le 2 then dcs_whitebeam_updatechan,state,1
         state.chans[2]:    if event.type le 2 then dcs_whitebeam_updatechan,state,2
         state.chans[3]:    if event.type le 2 then dcs_whitebeam_updatechan,state,3           
         else:
    endcase
    
    widget_control,event.handler,set_uvalue=state,/no_copy
end

pro dcs_whitebeam,group_leader=group_leader,workDir=workDir,dataDir=dataDir,DAVETool=DAVETool
    state = {group_leader:       0L, $   ;group leader
             tlb:                0L, $   ;top level base
             doneBut:            0L, $   ;exit
             ebarBut:            0L, $   ;show error bar
             xwavBut:            0L, $   ;x-axis wavelength
             xengBut:            0L, $   ;x-axis energy
             bmonBut:            0L, $   ;normalize to beam monitor
             duraBut:            0L, $   ;normalize to duration
             bm3But:      lonarr(3), $   ;[sample+empty,sample,empty]
             helpBut:            0L, $   ;help
             save1But:    lonarr(9), $   ;save plot1
             save2But:    lonarr(9), $   ;save plot2
             svsmBut :           0L, $   ;save sample raw data
             svemBut :           0L, $   ;save empty raw data
             fileText:      [0L,0L], $   ;[sample file text box,empty file text box]
             browBut:       [0L,0L], $   ;[sample file browse,empty file browse]
             chans:       lonarr(4), $   ;[pg(004),pg(002),lower cutoff,upper cutoff]
             showebar:           0b, $   
             xaxis:              0b, $   ;0-wavelength 1-energy
             norm:               0b, $   ;0-beam monitor 1-duration
             bm3:                0b, $   ;0-sample+empty 1-sample 2-empty
             pgind:   [-1,-1,-1,-1], $   ;
             dirs:          ['',''], $   ;[dataDir, workDir]
             files:         ['',''], $   ;[sample file, empty file]
             dataPtr:     ptrarr(2), $
             resultPtr:   ptr_new(), $
             plot1:       obj_new(), $   ;channel plot
             plot2:       obj_new(), $   ;transmission plot
             geom:    [0L,0L,0L,0L]  $   ;xsize,ysize,xoffset,yoffset
    }
    idl_version  = dm_to_number(!version.release)
    registerName = 'dcs_whitebeam'
    if xregistered(registerName) then begin   ;only allow one copy to be running at one time
       FORWARD_FUNCTION LookupManagedWidget
       id = LookupManagedWidget(registername)
       widget_control,id,/show,/realize,iconify=0
       return
    endif
    
    cd,current=current
    state.dirs = [current,current]
    defsysv,'!dave_defaults',exists=exists
    if exists then exists = ptr_valid(!dave_defaults)
    if (n_elements(dataDir) eq 0) and exists then begin
       if (file_test((*!dave_defaults).datDir,/directory,/read))[0] then dataDir = (*!dave_defaults).datDir
    endif 
    if n_elements(dataDir) ne 0 then state.dirs[0] = dataDir
    if (n_elements(workDir) eq 0) and exists then begin
       if (file_test((*!dave_defaults).workDir,/directory,/write))[0] then workDir = (*!dave_defaults).workDir
    endif
    if n_elements(workDir) ne 0 then state.dirs[1] = workDir
    if n_elements(group_leader) ne 0 then state.group_leader=group_leader
    
    bmpfiles      = ['open','mslice_save']+'.bmp'
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists
    for i=0,n_elements(bmpfiles)-1 do begin
      if exists then bmpfiles[i] = !DAVE_AUXILIARY_DIR+bmpfiles[i] $
      else bmpfiles[i] =  file_which(bmpfiles[i],/include_current_dir)
    endfor
    
    state.tlb     = widget_base(title='DCS White Beam Transmission',/col,kill_notify='dcs_whitebeam_Exit',/tlb_size_event,mbar=mbar,xpad=0,ypad=1,map=0,group_leader=state.group_leader)
    savetype      = ['PostScript','PDF','JPEG','GIF','TIFF','BMP','PNG','PPM','ASCII']+' File...'
    ;menu bar
    filemenu      = widget_button(mbar,value='File',/menu)
    spl1menu      = dm_widget_button(filemenu,value='Save BM3 Plot As',/menu,sensitive=0,imagefile=bmpfiles[1],/notchecked,uname='spl1menu')
    spl2menu      = dm_widget_button(filemenu,value='Save Transmission Plot As',/menu,sensitive=0,imagefile=bmpfiles[1],/notchecked,uname='spl2menu')
    for i=0,n_elements(savetype)-1 do begin
        if (i eq 1) and (idl_version lt 8.0) then continue ;PDF
        state.save1But[i] = widget_button(spl1menu,value=savetype[i],separator=(i eq n_elements(savetype)-1))
        state.save2But[i] = widget_button(spl2menu,value=savetype[i],separator=(i eq n_elements(savetype)-1))
    endfor
    state.svsmBut = dm_widget_button(filemenu,value='Save Sample File Data...',sensitive=0,imagefile=bmpfiles[1],/notchecked,/separator)
    state.svemBut = dm_widget_button(filemenu,value='Save Empty File Data...',sensitive=0,imagefile=bmpfiles[1],/notchecked)
    state.doneBut = widget_button(filemenu,value='Exit',/separator)
    optnmenu      = widget_button(mbar,value='Options',/menu)
    normmenu      = widget_button(optnmenu,value='Normalize To',/menu)
    xaxsmenu      = widget_button(optnmenu,value='Plot Transmission Vs',/menu)
    bm_3menu      = widget_button(optnmenu,value='BM3 Plot',/menu)
    state.bmonBut = dm_widget_button(normmenu,value='Beam Monitor (BM1)',set_button=(~state.norm))
    state.duraBut = dm_widget_button(normmenu,value='Duration',set_button=state.norm)
    state.xwavBut = dm_widget_button(xaxsmenu,value='Wavelength',set_button=(~state.xaxis))
    state.xengBut = dm_widget_button(xaxsmenu,value='Energy',set_button=state.xaxis)
    for i=0,2 do state.bm3But[i] = dm_widget_button(bm_3menu,value=(['Sample+Empty','Sample Only','Empty Only'])[i],set_button=(state.bm3 eq i))
    state.ebarBut = dm_widget_button(optnmenu,value='Show Error Bar',/separator,set_button=state.showebar)
    helpmenu      = widget_button(mbar,value='Help',/menu)
    state.helpBut = widget_button(helpmenu,value='Help')

    ;menu bar separator for windows system
    if !version.os_family eq 'Windows' then begin
       mbar_sep   = widget_label(state.tlb,sensitive=0,/dynamic_resize,value=' ',scr_ysize=5)
    endif
    
    line = widget_base(state.tlb,/row,/align_left)
    void = widget_label(line,value='sample file:',/align_right)
    geom = widget_info(void,/geometry)
    widget_control,void,scr_xsize=geom.scr_xsize+5
    void = widget_label(line,value=' ')
    state.fileText[0] = widget_text(line,value='',/all_events,scr_xsize=200,/editable)
    state.browBut[0] = widget_button(line,value=bmpfiles[0],/bitmap,/dynamic_resize)
    void = widget_label(line,value=' ',scr_xsize=20)
    void = widget_label(line,value='PG (004) channel:')
    void = widget_label(line,value=' ')
    state.chans[0] = widget_text(line,value='',scr_xsize=80,/all_events,/editable)
    void = widget_label(line,value=' ',scr_xsize=10)
    lcoc = widget_label(line,value='lower cutoff channel:')
    xsiz = (widget_info(lcoc,/geometry)).scr_xsize
    void = widget_label(line,value=' ')
    state.chans[2] = widget_text(line,value='',scr_xsize=80,/all_events,/editable)
    geom1 = widget_info(line,/geometry)
    line = widget_base(state.tlb,/row,/align_left)
    void = widget_label(line,value='empty file:',/align_right)
    widget_control,void,scr_xsize=geom.scr_xsize+5 
    void = widget_label(line,value=' ')
    state.fileText[1] = widget_text(line,value='',/all_events,scr_xsize=200,/editable)   
    state.browBut[1] = widget_button(line,value=bmpfiles[0],/bitmap,/dynamic_resize)
    void = widget_label(line,value=' ',scr_xsize=20)
    void = widget_label(line,value='PG (002) channel:')
    void = widget_label(line,value=' ')
    state.chans[1] = widget_text(line,value='',scr_xsize=80,/all_events,/editable)
    void = widget_label(line,value=' ',scr_xsize=10)
    ucoc = widget_label(line,value='upper cutoff channel:')
    xsiz = xsiz>((widget_info(ucoc,/geometry)).scr_xsize)
    widget_control,ucoc,scr_xsize=xsiz
    widget_control,lcoc,scr_xsize=xsiz
    void = widget_label(line,value=' ')
    state.chans[3] = widget_text(line,value='',scr_xsize=80,/all_events,/editable)
    if float(!version.release) ge 5.6 then for i=0,1 do widget_control,state.browBut[i],tooltip='Click to select a file.' 
    void = widget_label(state.tlb,value=' ')
    if idl_version ge 5.6 then scr_size = get_screen_size() else device,get_screen_size=scr_size
    xsize = 800<(fix((scr_size[0]-90)/2))>geom1.scr_xsize
    ysize = fix(5./9*xsize)<((scr_size[1]-100)/2)
    fsize = 12>(round(13.*xsize/800))
    state.plot1 = obj_new('dm_plot',xsize=xsize,ysize=ysize,widgetbase=state.tlb,background='white',group_leader=state.tlb,/compound,/isolatin1,fsize=fsize,/legdshowoutline,/legdshowfill,legdpos=[0.63,0.95],/showcursorpos,ctxtfsize=10,path=state.dirs[1])
    state.plot2 = obj_new('dm_plot',xsize=xsize,ysize=ysize,widgetbase=state.tlb,background='white',group_leader=state.tlb,/compound,/isolatin1,fsize=fsize,ctxtfsize=10,path=state.dirs[1])
    
    dm_center_kid,state.tlb,state.group_leader,/side
    widget_control,state.tlb,/update,/realize,/map
    geom = widget_info(state.tlb,/geometry)
    state.geom = [geom.scr_xsize,geom.scr_ysize,geom.xoffset,geom.yoffset]
    widget_control,state.tlb,set_uvalue=state 
    xmanager,registerName,state.tlb,cleanup='dcs_whitebeam_Exit',event_handler='dcs_whitebeam_event',/no_block
end