; docformat = 'rst'

;+
; Object representing an HDF 5 group.
;
; :Categories:
;   file i/o, hdf5, sdf
;
; :Requires:
;   IDL 8.0
;
; :Author:
;   Michael Galloy
;-


;+
; Get properties
;
; :Keywords:
;   _ref_extra : out, optional, type=keywords
;     properties of `MGffH5Base`
;-
pro mgffh5group::getProperty, _ref_extra=e
  compile_opt strictarr

  if (N_elements(e) gt 0L) then self->Mgffh5base::getproperty, _extra=e
end


;+
; Open an HDF 5 group.
;
; :Private:
;
; :Keywords:
;   error : out, optional, type=long
;     error code: 0 for none
;-
pro mgffh5group::_open, error=error
  compile_opt strictarr

  Catch, error
  if (error ne 0L) then begin
    Catch, /cancel
    Return
  endif

  self.parent->Getproperty, identifier=parent_id
  self.id = H5g_open(parent_id, self.name)
end


;+
; Close an HDF 5 group.
;
; :Private:
;
; :Keywords:
;   error : out, optional, type=long
;     error code: 0 for none
;-
pro mgffh5group::_close, error=error
  compile_opt strictarr

  Catch, error
  if (error ne 0L) then begin
    Catch, /cancel
    Return
  endif

  H5g_close, self.id
end


;+
; Get object info for child object inside group.
;
; :Private:
;
; :Returns:
;   `H5F_STAT` structure
;
; :Params:
;   name : in, required, type=string
;     name of child object
;
; :Keywords:
;   error : out, optional, type=long
;     error code: 0 for none
;-
function mgffh5group::_statObject, name, error=error
  compile_opt strictarr

  Catch, error
  if (error ne 0L) then begin
    Catch, /cancel
    Return, -1
  endif

  for i = 0L, H5g_get_num_objs(self.id) - 1L do begin
    ; find first case-insensitive match
    objName = H5g_get_obj_name_by_idx(self.id, i)
    if (Strcmp(objName, name, /fold_case)) then begin
      name = objName
      break
    endif
  endfor

  Return, H5g_get_objinfo(self.id, name)
end


;+
; Output for `HELP` for group.
;
; :Returns:
;   string
;
; :Params:
;   varname : in, required, type=string
;     name of variable containing group object
;-
function mgffh5group::_overloadHelp, varname
  compile_opt strictarr

  type = 'H5Group'
  self->Getproperty, fullname=fullname
  specs = String(fullname, format='(%"%s")')
  Return, self->Mgffh5base::_overloadhelp(varname, type=type, specs=specs)
end


;+
; Output for `PRINT` for group.
;
; :Private:
;
; :Returns:
;   string
;-
function mgffh5group::_overloadPrint
  compile_opt strictarr

  self->_Open, error=error
  if (error ne 0L) then Message, 'invalid HDF5 file'

  nmembers = H5g_get_num_objs(self.id)
  names = Strarr(nmembers)
  types = Strarr(nmembers)
  for g = 0L, nmembers - 1L do begin
    names[g] = H5g_get_obj_name_by_idx(self.id, g)
    info = H5g_get_objinfo(self.id, names[g])
    types[g] = info.type
  endfor

  ; TODO: show attributes as well

  listing = mg_strmerge('  ' + types + ' ' + names)

  Return, String(self.name, mg_newline(), listing, format='(%"GROUP %s%s%s")')
end


;+
; Handles accessing groups/variables, particularly those with case-sensitive
; names or spaces/other characters in their names.
;
; :Private:
;
; :Returns:
;   variable object
;
; :Examples:
;   For example::
;
;     h = mg_h5(file_which('hdf5_test.h5'))
;     d = h['2D int array']
;
; :Params:
;   isRange : in, required, type=lonarr(8)
;     indicates whether the i-th parameter is a index range or a scalar/array
;     of indices
;   ss1 : in, required, type=long/lonarr
;     scalar subscript index value, an index array, or a subscript range
;   ss2 : in, optional, type=any
;     not used
;   ss3 : in, optional, type=any
;     not used
;   ss4 : in, optional, type=any
;     not used
;   ss5 : in, optional, type=any
;     not used
;   ss6 : in, optional, type=any
;     not used
;   ss7 : in, optional, type=any
;     not used
;   ss8 : in, optional, type=any
;     not used
;-
function mgffh5group::_overloadBracketsRightSide, isRange, $
  ss1, ss2, ss3, ss4, $
  ss5, ss6, ss7, ss8
  compile_opt strictarr
  On_error, 2

  objInfo = self->_Statobject(ss1, error=error)
  if (error ne 0L) then Message, String(ss1, format='(%"object %s not found")')

  ; TODO: search existing children before creating a new one

  case objInfo.type of
    'GROUP': result = Obj_new('MGffH5Group', parent=self, name=ss1)
    'DATASET': result = Obj_new('MGffH5Dataset', parent=self, name=ss1)
    'TYPE': ; TODO: implement
    'LINK': ; TODO: implement
    'UNKNOWN': Message, String(ss1, format='(%"object %s unknown")')
    else: begin
      ; TODO: check for attribute
      Message, String(ss1, format='(%"object %s unknown")')
    end
  endcase

  self.children->Add, result

  if (N_elements(ss2) gt 0L) then begin
    Return, result->_Overloadbracketsrightside(isRange[1:*], ss2, ss3, ss4, ss5, ss6, ss7, ss8)
  endif else begin
    Return, result
  endelse
end


;= lifecycle methods

;+
; Free resources.
;-
pro mgffh5group::Cleanup
  compile_opt strictarr

  Obj_destroy, self.children
  self->_Close
end


;+
; Create a group object.
;
; :Returns:
;   1 for success, 0 for failure
;
; :Keywords:
;   _extra : in, optional, type=keywords
;     keywords to `MGffH5Base::init`
;-
function mgffh5group::init, _extra=e
  compile_opt strictarr
  On_error, 2

  if (~self->Mgffh5base::init(_extra=e)) then Return, 0

  self.children = Obj_new('IDL_Container')

  self->_Open, error=error
  if (error ne 0L) then Message, 'invalid HDF5 group'

  Return, 1
end


;+
; Define instance variables and class inheritance.
;
; :Fields:
;   children
;     `IDL_Container` containing children groups/datasets
;-
pro mgffh5group__define
  compile_opt strictarr

  define = { Mgffh5group, inherits Mgffh5base, $
    children: Obj_new() $
  }
end