; $Id$
;###############################################################################
;
; NAME:
;   dave
;
; PURPOSE:
;   Define useful system variables, units and units convertion factors
;   for the DAVE framework.
;
; CATEGORY:
;   DAVE framework
;
; AUTHORS:
;   John R.D. Copley
;   Sungil Park
;
;   NIST Center for Neutron Research
;   100 Bureau Drive, Gaithersburg, MD 20899
;   United States
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################



;===============================================================================
; daveSysVars.pro
;
; Declaration of some important system variables
; Written by John Copley
; Modified by Sungil Park
;	System Variables in SI units.
;	Unit conversion routine: dave_unit_conversion().
;
;*******************************************************************************
;23456789 123456789 123456789 123465798 123456789 123456789 123465789 123456789
;       10        20        30        40        50        60        70        80
;*******************************************************************************
; Dave Unit Conversion
;
; Usage:
;	converted_data = $
;		original_data * dave_unit_conversion('energy', from='eV', to='meV')
;
function dave_unit_conversion ,physical_quantity, from=original_unit, to=destination_unit

case physical_quantity of
    'energy': $
      begin
        
        units = $
          [ 'J' $               ; Default
            , 'eV' $
            , 'meV' $
            , 'ueV' $           ; micro eV
          ]
        
        factors = $
          [ 1D $
            , !dave_1_over_e $  ; eV
            , !dave_1_over_e * 1d3 $ ; meV
            , !dave_1_over_e * 1d6 $ ; ueV
          ]
        
    end
    'wavevector': $
      begin
        units = $
          [ 'm-1' $             ; Default
            , 'nm-1' $
            , 'A-1' $
          ]
        
        factors = $
          [ 1D $
            , 1d-9 $
            , 1d-10 $
          ]
    end
    'wavelength': $
      begin
        units = $
          [ 'm' $               ; Default = meters
            , 'nm' $            ; nanometers
            , 'A' $             ; Angstroms
          ]
        factors = $
          [ 1d $
            , 1d9 $
            , 1d10 $
          ]
    end
    'time': $
      begin
        units = $
          [ 's' $               ; Default = seconds
            , 'ms' $            ; miliseconds
            , 'us' $            ; microseconds
            , 'ns' $            ; nanoseconds
            , 'ps' $            ; picoseconds
          ]
        factors = $
          [ 1D $
            , 1d3 $
            , 1d6 $
            , 1d9 $
            , 1d12 $
          ]
    end
    else: $
      begin
        message $
          , 'Unknown Physical Quantity. Returning ZERO!'$
          , /continue
        return, 0D
    end
endcase

;; If the keyword parameters are not given,
;; the default SI units are assumed.
;; If the given keyword parameter cannot be found,
;; 0 is returned.
if n_elements( original_unit ) ne 1 then begin
    from = 0
endif else begin
    from = where( units eq original_unit, count )
    if count eq 0 then begin    ; No match
        message $
          , 'Unknown Original Unit. Returning ZERO!'$
          , /continue
        return, 0D
    endif
endelse

if n_elements( destination_unit ) ne 1 then begin
    to = 0
endif else begin
    to = where( units eq destination_unit, count )
    if count eq 0 then begin    ; No match
        message $
          , 'Unknown Destination Unit. Returning ZERO!'$
          , /continue
        return, 0D
    endif
endelse

return, factors[ to ] / factors[ from ]

end


;*******************************************************************************
;23456789 123456789 123456789 123465798 123456789 123456789 123465789 123456789
;       10        20        30        40        50        60        70        80
;*******************************************************************************
pro daveSysVars

DEBUG = 0

;; No need to run twice.
defsysv, '!dave_planck', exists = exists
if exists then return

;; Fundamental Physical Constants
;; as shown in NIST website. - Sungil
;;
;; Planck Constant, h
defsysv, '!dave_planck', 6.62606876d-34, 1 ; J s
defsysv, '!dave_planck_eV_s', 4.13566727d-15, 1 ; eV s

;; Planck Constant over 2 Pi, hbar
defsysv, '!dave_hbar', 1.054571596d-34, 1 ; J s
defsysv, '!dave_hbar_eV_s', 6.58211889d-16, 1 ; eV s

;; Neutron Mass
defsysv, '!dave_nmass', 1.67492716d-27, 1 ; kg
defsysv, '!dave_nmass_J', 1.50534946d-10, 1 ; J
defsysv, '!dave_nmass_eV', 939.565330d6, 1 ; eV

;; Boltzmann Constant
defsysv, '!dave_boltzmann', 1.3806503d-23, 1 ; J K-1
defsysv, '!dave_boltzmann_eV_over_K', 8.617342d-5, 1 ; eV K-1
defsysv, '!dave_boltzmann_Hz_over_K', 2.0836644d10, 1 ; Hz K-1 ( k/h )
defsysv, '!dave_boltzmann_1_over_m_K', 69.50356, 1 ; m-1 K-1 ( k/hc )

;; Elementary Charge
defsysv, '!dave_e', 1.602176462d-19, 1 ; C
defsysv, '!dave_e_over_h', 2.417989491d14, 1 ; A J-1
defsysv, '!dave_1_over_e',  6.24150974d18, 1 ; C-1

;; Speed of Light in Vacuum, c
defsysv, '!dave_c', 299792458d, 1 ; m s-1
defsysv, '!dave_c_A_over_s', 299792458d10, 1 ; A s-1

;; Frequently Used Symbols
defsysv, '!dave_hbarSym',       '!3!sh!r!e/!n'
defsysv, '!dave_omegaSym',      '!4x!3'
defsysv, '!dave_angstromSym',   '!3!sA!r!u!9 %!3!n'
defsysv, '!dave_invAngstromSym','!3!sA!r!u!9 %!3!n!E-1!N'
defsysv, '!dave_microSym',      '!4l!3'
defsysv, '!dave_inversenmSym',  '!3nm!u-1!n'

color = {color, $               ; name
         red:   [255,0,255,  0,  0,  0,255,255,152,128,128,128,255,178,255],$ ; red values
         green: [255,0,  0,255,  0,255,  0,255,120,128,128,128, 54,192,128],$ ; green values
         blue:  [255,0,  0,  0,255,255,255,  0,103,220, 60,128, 54,220,128] $ ; blue values
        }

defsysv, '!dave_colors', color, 1

; Create a system variable for a window (RMD-Feb 04)
defsysv,'!dave_e_window',ptr_new(/allocate_heap)

;
defsysv,'!operating_system',!VERSION.OS_FAMILY,1
defsysv,'!windows',(!operating_system eq "Windows"),1
defsysv,'!unix',(!operating_system eq "unix"),1
defsysv,'!macos',(!operating_system eq "MacOS"),1

screensize=GET_SCREEN_SIZE()
defsysv,'!screensize',screensize,1

defsysv,'!delimiter',path_sep()

; Figure out what OS is running and define the concept of home
; directory so that the defaults file doesn't get spread all over
; the hard disk

home_dir = getenv("DAVE_DEFAULTS_DIR")

if (strlowcase(!version.os_family) eq "windows" ) then begin
    
    if (strtrim(home_dir,2) eq "") then begin
        home_dir = getenv("HOME")
        
        if (strtrim(home_dir,2) eq "") then begin
            home_dir = getenv('HOMEDRIVE')+getenv('HOMEPATH')+path_sep()
        endif

        if (strtrim(home_dir,2) eq "") then begin
            home_dir = "C:"+!delimiter
        endif 

        if (strtrim(home_dir,2) eq "") then begin
            home_dir = "C:"+!delimiter
        endif 
    endif else begin
        home_dir = home_dir+!delimiter
    endelse
    
endif

if (strlowcase(!version.os_family) eq "unix" ) then begin
    
    if (strtrim(home_dir,2) eq "") then begin
        home_dir = getenv("HOME")+!delimiter
    endif else begin
        home_dir = home_dir+!delimiter
    endelse
    
endif

if (strlowcase(!version.os_family) eq "macos" ) then begin
    
    ;; find the root directory first
    colon_loc = strpos(!path,':') ;
    root_dir = strmid(!path,0,colon_loc) ;
    
    
    if (strtrim(home_dir,2) eq "") then begin
        home_dir = getenv("HOME")
        if (strtrim(home_dir,2) eq "") then begin
            home_dir = root_dir+":System Folder:Preferences:"
            prefs_dir_exists = file_test(home_dir,/directory,/write) ;
            if (prefs_dir_exists eq 0) then begin
                home_dir = root_dir ;
            endif
        endif else begin
            home_dir = home_dir+!delimiter
        endelse
    endif else begin
        home_dir = home_dir+!delimiter
    endelse
    
endif

defaults_file = home_dir+path_sep()+".dave_defs.sav" ;
defsysv,'!defaults_file',defaults_file
defsysv,'!home_dir',home_dir

defsysv, '!debug',DEBUG
defsysv, '!ftkwwFlag',0

if (!debug) then begin
    print, "OS: ",!version.os_family
    print, "Home Directory: ",home_dir
    print, "Defaults File: ",!defaults_file
endif


print,'DAVE System variables have been defined'

return
end

