; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;

;----------------------------------------------------------------------------
;+
; :Returns:
;   a string giving the filename of the external file
;
; :Arguments:
;   filename: in, required, String
;     a string giving the filename
;   item: in, required, ASDF_NDARRAY
;     a ndarray object that holds the data we intend to write.
;   nexternal: in, required, Integer
;     a running counter used to differentiate filenames for the external writes
;
;-
function asdf_write_external, filename, item, nexternal
  compile_opt idl2, hidden
  asdf = ASDF_File()
  externalfile = filename
  if (externalfile.endsWith('.asdf', /fold_case)) then begin
    externalfile = externalfile.remove(-5)
  endif
  externalfile += `${nexternal,'%04d'}.asdf`
  openw, lun, externalfile, /get_lun
  printf, lun, asdf
  ; signals end of metadata.
  printf, lun, '...'
  free_lun, lun, /force
  item._WriteData, externalfile
  return, file_basename(externalfile)
end


;----------------------------------------------------------------------------
;+
; :Description:
;   Add the ASDF_NDArray to our internal list (to be written later),
;   or write out externally.
;
; :Arguments:
;   item: in, required, ASDF_NDArray
;     an ASDF_NDArray item to output to the file
;   filename: in, required, String
;     a string giving the filename
;   items: bi, required, Structure<any>
;     a list of ASDF-ND-array objects that make up our file
;   nexternal: bi, required, Integer
;     a running counter used to differentiate filenames for the external writes
;
;-
pro asdf_collate_and_write, item, filename, items, nexternal
  compile_opt idl2, hidden

  ; Inline array?
  if (~item.hasKey('source')) then return
  
  source = item.source

  if (isa(source, /number)) then begin
    if (~isa(items)) then items = list()
    item.source = items.length
    items.add, item
    return
  endif

  if (isa(source, /string)) then begin
    ; Start our external filename counter, for files 0000, 0001, 0002,...
    if (~isa(nexternal)) then nexternal = 0LL
    item.source = asdf_write_external(filename, item, nexternal)
    nexternal++
  endif
end


;----------------------------------------------------------------------------
;+
; :Description:
;   Retrieve our data as a list of ASDF_NDarrays.
;   Writes out any external data, so we don't have to cache the data.
;
; :Arguments:
;   node: in, required, Hash<any> | List<any>
;     a list or hash containing (possibly) some ASDF_NDArray nodes
;   filename: in, required, String
;     a string giving the filename
;   items: bi, required, List<Structure<any>>
;     a list of ASDF-ND-array objects that make up our file
;   nexternal: in, required, Integer
;     a running counter used to differentiate filenames for the external writes
;
;-
pro asdf_write_walk_metadata, node, filename, items, nexternal
  compile_opt idl2, hidden

  if (isa(node, 'list') || isa(node, 'hash')) then begin
    foreach item, node, key do begin
      if (isa(item, 'ASDF_NDArray')) then begin
        asdf_collate_and_write, item, filename, items, nexternal
      endif else if (isa(item, /scalar) && isa(item, /complex)) then begin
        node[key] = yaml_value(item, tag='!core/complex-1.0.0')
      endif else begin
        asdf_write_walk_metadata, item, filename, items, nexternal
      endelse
    endforeach
  endif
end


;----------------------------------------------------------------------------
;+
; :Description:
;   Write out all of our internal data to the end of our file.
;
; :Arguments:
;   filename: in, required, String
;     a string giving the filename
;   items: in, required, List<Structure<any>>
;     of ASDF-ND-array objects that make up our file. Ready to be written
;
;-
pro asdf_write_internal_data, filename, items
  compile_opt idl2, hidden
  foreach item, items do begin
    item._WriteData, filename
  endforeach
end


;----------------------------------------------------------------------------
;+
; :Description:
;   The asdf_write function takes an ASDF_File object and
;   writes out an ASDF file.
;
; :Arguments:
;   filename: in, required, String
;     A string giving the full filename where the data should be written.
;   asdf: in, required, ASDF_File
;     *Asdf* must be an ASDF_File object containing the data.
;
; :Keywords:
;   debug: in, optional, Boolean
;     Set this keyword to disable error catching. Undocumented.
;
; :Author:
;   CT, March 2023.
;
;-
pro asdf_write, filename, asdf, DEBUG=debug

  compile_opt idl2, hidden
  lun = 0
  if (~keyword_set(debug)) then begin
    on_error, 2
    catch, ierr
    if (ierr ne 0) then begin
      catch, /cancel
      if (lun ne 0) then free_lun, lun, /force
      msg = (!error_state.msg).Replace('YAML_PARSE_INTERNAL: ', '')
      message, msg
    endif
  endif

  catch, err
  if (err eq 0) then begin
    idlsoftware = YAML_Map('name', 'IDL', 'version', `${!version.release}`)
    idlsoftware.tag = '!core/software-1.0.0'
    idlsoftware.style = 'flow'
    asdf['history', 'extensions', 0, 'software'] = idlsoftware
  endif
  catch, /cancel

  asdf_write_walk_metadata, asdf, filename, items
  
  ; Write out the final asdf file. This happens regardless of whether the data is 
  ; internal or external.
  yaml = yaml_serialize(asdf)
  openw, lun, filename, /get_lun
  printf, lun, yaml
  ; signals end of metadata.
  printf, lun, '...'
  free_lun, lun, /force
  
  ;Write the data we collected at the end of the file.
  ;This is the binary at the bottem of the file.
  if (isa(items)) then begin
    asdf_write_internal_data, filename, items
  endif

end