; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.

;+
; NAME:
; savefile_cleanup
;
; PURPOSE:
; Scan a savefile for routines that have been ported from PRO code
; to C code in idl.dll, removing these deprecated methods.
;
; CATEGORY:
; Maintenance helper.
;
; CALLING SEQUENCE:
; savefile_cleanup, filename
;
; INPUTS:
; filename = the name of the save file to process
;
; OUTPUTS:
; New save file with deprecated routines removed.
;
; SIDE EFFECTS:
; The original save file is renamed with a .bak extension.
;
; RESTRICTIONS:
; None.
;
; PROCEDURE:
; Use IDL_Savefile class to inspect the save file for its symbol manifest.
; Compare the contained symbols with list of known deprecated routines.
; If no deprecated routines, then do nothing and exit.
; Rename the current file with a .bak extension.
; Start an IDL_IDLBridge to do the work in a clean runtime.
; Use IDL_Savefile to conditionally restore the non-deprecated routines.
; Create a new save file containing only the restored routines.
;
; MODIFICATION HISTORY:
;   BJG, May 2021 - Initial version for IDL 8.8.1
;-

function savefile_cleanup_skip_classes
  compile_opt idl2, hidden


  classes = ['COLLECTION', $
             'DICTIONARY', $
             'HASH', $
             'LIST', $
             'ORDEREDHASH']

  return, classes
end


function savefile_cleanup_skip_functions
  compile_opt idl2, hidden

  sysFunc = Routine_Info(/Functions, /System)

  classes = savefile_cleanup_skip_classes()

  ret = List('REVERSE')

  foreach c, classes do begin
    w = Where((sysFunc eq c) + $
              sysFunc.StartsWith(c+'::') + $
              sysFunc.StartsWith(c+'_'), count)
    if (count gt 0) then begin
      ret.Add, sysFunc[w], /Extract
    endif
  endforeach

  return, (ret.ToArray()).Sort()
end


function savefile_cleanup_skip_procedures
  compile_opt idl2, hidden

  sysProc = Routine_Info(/System)

  ret = List()

  classes = savefile_cleanup_skip_classes()
  foreach c, classes do begin
    w = Where(sysProc.StartsWith(c+'::') + $
              sysProc.StartsWith(c+'__DEFINE'), count)
    if (count gt 0) then begin
      ret.Add, sysProc[w], /Extract
    endif
  endforeach

  return, (ret.ToArray()).Sort()
end


pro savefile_cleanup, filenameIn, VERBOSE=verbose
  compile_opt idl2

  if (~ISA(filenameIn, /Scalar, /String)) then begin
    Message, 'Filename must be a scalar string.'
  endif

  if (~File_Test(filenameIn, /Read, /Regular)) then begin
    Message, 'Invalid filename: ' + filenameIn +'.'
  endif

  ; convert any relative paths to absolute path
  info = File_Info(filenameIn)
  filename = info.Name

  Catch, iErr
  if (iErr eq 0) then begin
    sv = IDL_Savefile(filename)
  endif else begin
    Catch, /Cancel
    Message, 'Not a valid save file: ' + filename + '.'
  endelse

  Catch, /Cancel

  ; check if this is a data save file or routine save file
  contents = sv.Contents()
  if ((contents.n_procedure eq 0) && (contents.n_function eq 0)) then begin
    Message, 'File is a data save file: ' + filename + '.'
  endif

  ; get the description, to put into new save file
  description = contents.Description

  allFuncNames = sv.Names(/Function)
  if (allFuncNames[0] ne '') then begin
    numFunc = allFuncNames.Diff(savefile_cleanup_skip_functions(), Removed=goodFuncs)
    if (goodFuncs[0] eq '') then begin
      goodFuncs = !null
    endif else begin
      numFunc = allFuncNames.Diff(goodFuncs, Removed=badFuncs)
      if (badFuncs[0] eq '') then begin
        badFuncs = !null
      endif
    endelse
  endif else begin
    goodFuncs = !null
  endelse

  allProcNames = sv.Names(/Procedure)
  if (allProcNames[0] ne '') then begin
    numProc = allProcNames.Diff(savefile_cleanup_skip_procedures(), Removed=goodProcs)
    if (goodProcs[0] eq '') then begin
      goodProcs = !null
    endif else begin
      numProc = allProcNames.Diff(goodProcs, Removed=badProcs)
      if (badProcs[0] eq '') then begin
        badProcs = !null
      endif
    endelse
  endif else begin
    goodProcs = !null
  endelse

  ; close the save file  
  sv = 0

  ; if no non-deprecated symbols, announce as such and return
  if (~ISA(goodFuncs) && ~ISA(goodProcs)) then begin
    print, 'Save file has only deprecated symbols in it.'
    return
  endif

  ; if no symbols to remove, announce as such and return
  if (~ISA(badFuncs) && ~ISA(badProcs)) then begin
    print, 'Save file has no non-deprecated symbols in it.'
    return
  endif

  ; rename the save file to a backup filename
  bakFilename = filename + '.bak'
  File_Move, filename, bakFilename, /Overwrite
  print, 'Backed up save file to ' + bakFilename + '.'

  if (Keyword_Set(verbose)) then begin
    if (ISA(goodProcs)) then begin
      foreach p, badProcs do begin
        Message, 'Removed procedure: ' + p + '.', /Info
      endforeach
    endif
    if (ISA(goodFuncs)) then begin
      foreach f, badFuncs do begin
        Message, 'Removed function: ' + f + '.', /Info
      endforeach
    endif
  endif

  ; create an IDL_IDLBridge to do the work in a clean environment
  oBridge = IDL_IDLBridge()
  ; set current working directory, in case filename is relative
  cd, Current=pwd
  oBridge.Execute, 'cd, ' + pwd
  oBridge.Execute, 'sv = IDL_SaveFile("' + bakFilename + '")'

  ; restore all the non-deprecated symbols 
  if (ISA(goodFuncs)) then begin
    oBridge.SetVar, 'goodFuncs', goodFuncs
    oBridge.Execute, 'sv.Restore, goodFuncs, /Function, /No_Compile'
  endif
  if (ISA(goodProcs)) then begin
    oBridge.SetVar, 'goodProcs', goodProcs
    oBridge.Execute, 'sv.Restore, goodProcs, /Procedure, /No_Compile'
  endif

  ; save the output save file
  cmd = String(Format='Save, /Routine, Filename=\"%s\"', filename)
  if (description.Strlen() gt 0) then begin
    cmd += String(Format=', Description=\"%s\"', description)
  endif
  oBridge.Execute, cmd
  print, 'Saved clean save file to ' + filename + '.'

  ; clean up the bridge
  b = 0
end
