; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;----------------------------------------------------------------------------
; Purpose:
;   This file implements the IDLitReadShapefile class.
;
;---------------------------------------------------------------------------
; Lifecycle Routines
;---------------------------------------------------------------------------
; Purpose:
;   The constructor of the object.
;
; Arguments:
;   None.
;
; Keywords:
;   All keywords to superclass.
;
function IDLitReadShapefile::Init, _REF_EXTRA=_extra

    compile_opt idl2, hidden

    ; Init superclass
    if (~self->IDLitReader::Init("shp", $
        NAME='ESRI Shapefile', $
        DESCRIPTION="ESRI Shapefile (shp)", $
        ICON='drawing', $
        _EXTRA=_extra)) then $
        return, 0

    self->RegisterProperty, 'COMBINE_ALL', /BOOLEAN, $
        NAME='Combine all shapes', $
        DESCRIPTION='Combine all shapes into a single data object'

    self->RegisterProperty, 'ATTRIBUTE_NAME', $
        ENUMLIST=['<Shape index>'], $
        NAME='Name attribute', $
        DESCRIPTION='Attribute to use for the shape name'

    ; Combine all shapes by default.
    self._combineAll = 1

    return, 1
end


;---------------------------------------------------------------------------
pro IDLitReadShapefile::GetProperty, $
    COMBINE_ALL=combineAll, $
    ATTRIBUTE_NAME=attributeName, $
    LIMIT=limit, $
    _REF_EXTRA=_extra

    compile_opt idl2, hidden

    if ARG_PRESENT(combineAll) then $
        combineAll = self._combineAll

    if ARG_PRESENT(attributeName) then $
        attributeName = self._attributeIndex

    if ARG_PRESENT(limit) then $
        limit = self._limit

    if (N_ELEMENTS(_extra) gt 0) then $
        self->IDLitReader::GetProperty, _EXTRA=_extra
end


;---------------------------------------------------------------------------
pro IDLitReadShapefile::SetProperty, $
    COMBINE_ALL=combineAll, $
    ATTRIBUTE_NAME=attributeName, $
    LIMIT=limit, $
    _REF_EXTRA=_extra

    compile_opt idl2, hidden

    if (N_ELEMENTS(combineAll) eq 1) then begin
        self._combineAll = combineAll
        self->SetPropertyAttribute, 'ATTRIBUTE_NAME', $
            SENSITIVE=~self._combineAll
    endif

    if (N_ELEMENTS(attributeName) eq 1) then $
        self._attributeIndex = attributeName

    if (N_ELEMENTS(limit) gt 0) then $
        self._limit = limit

    if (N_ELEMENTS(_extra) gt 0) then $
        self->IDLitReader::SetProperty, _EXTRA=_extra
end


;---------------------------------------------------------------------------
; IDLitReadShapefile::_GetNameAttribute
;
; Purpose:
;   Given a shape object, returns the best guess for the index of the
;   attribute to be used for the name of each shape.
;   Returns -1 if no useable attribute exists.
;
function IDLitReadShapefile::_GetNameAttribute, oShape

    compile_opt idl2, hidden

    oShape->GetProperty, N_ATTRIBUTES=nAttr
    if (~nAttr) then $
        return, -1

    oShape->GetProperty, ATTRIBUTE_INFO=attrInfo, $
        N_ENTITIES=nEntity

    ; CT, Jan 2024: Hack for our wb_countries_2020 file, added in IDL 9.1
    match = (where(attrInfo.name eq 'NAME_EN'))[0]
    if (match ge 0) then return, match

    ; See if we have a string attribute, and assume this is a name.
    strIndex = WHERE(attrInfo.type eq 7, nstr)
    if (~nstr) then $
        return, -1

    widest = MAX(attrInfo[strIndex].width, loc)
    nameIndex = strIndex[loc]

    ; If we have just one entity then we can go ahead
    ; and use the attribute.
    if (nEntity eq 1) then $
        return, nameIndex

    ; Otherwise, make sure that our supposed name attribute
    ; actually varies per entity.
    attr0 = oShape->GetAttributes(0)
    attr1 = oShape->GetAttributes(1)
    return, (attr0.(nameIndex) ne attr1.(nameIndex)) ? nameIndex : -1

end


;---------------------------------------------------------------------------
; This logic is to find the continent of Antartica which is screwed up
; in the shapefile.  They add a bunch of extraneous points between the
; pole and the international dateline (both + and - 180 degs), to make
; a simple plot of the vertices look good on a cylindrical map,
; centered on the prime meridian.  This screws up our polygon
; filling and adds an extraneous line on Antarctica. We then
; remove the extraneous points.
;
function IDLitReadShapefile::_FixAntarctica, pVert, $
    pStart, nSubVert, index

    compile_opt idl2, hidden

    xy = (*pVert)[*, pStart:pStart+nSubvert-1]

    ; Make sure we have the South Pole.
    if total(xy[1,*] eq -90) lt 2 then $
        goto, bailout

    ; Need to make counterclockwise for fill to work correctly.
    xy = REVERSE(xy, 2)

    ; Find the points which run along the dateline.
    bad = WHERE((ABS(ABS(xy[0, *]) - 180) lt 0.001d) or (xy[1,*] eq -90), nbad)

    ; IDL-68972: For the GSHHS_h_L1 shapefile, the first Antarctica point
    ; is at 180E, 78S. This is actually a good point,
    ; so remove it from the "bad" list.
    if (nbad gt 2 && bad[0] eq 0) then begin
      bad = bad[1:*]
      nbad--
    endif

    if (nbad lt 2) then $
      goto, bailout

    ; Tweak the points at the beginning/end of the cut so
    ; they don't lie exactly on +/-180.
    xy[0, *] = -179.99d > xy[0, *] < 179.99d

    if (bad[1] lt nSubVert - 10) then begin
      ; If the bad points are in the middle of the shape,
      ; move them to the end.
      xy = [[xy[*, bad[-1]:*]], [xy[*, 0:bad[-1]-1]]]
      nbad = bad[-1] - bad[0] - 1
    endif else begin
      ; IDL-68972: For the GSHHS_h_L1 shapefile, the bad Antarctica points
      ; are at the end of the array. Just drop them from the connectivity.
      nbad = nSubVert - bad[0]
    endelse

    ; Put our modified data back into the original vertext array.
    (*pVert)[0, pStart] = xy

    ; All of the bad points should now be at the end of the vertex list.
    ; Return a connectivity array containing only the good points.
    ngood = nSubVert - nbad
    return, [ngood, LINDGEN(ngood) + pStart + index]

bailout:
    ; Just return a regular connectivity list.
    return, [nSubVert, LINDGEN(nSubVert) + pStart + index]

end


;---------------------------------------------------------------------------
; IDLitReadShapefile::GetData
;
; Purpose:
; Read the image file and return the data in the data object.
;
; Parameters:
;
; Returns 1 for success, 0 otherwise.
;
function IDLitReadShapefile::GetData, oData

    compile_opt idl2, hidden

    strFilename = self->GetFilename()

    oShape = OBJ_NEW('IDLffShape')

    unableRead = "Unable to read file '" + strFilename + "'"
    if (~oShape->Open(strFilename)) then begin
        self->SignalError, unableRead, SEVERITY=2
        OBJ_DESTROY, oShape
        return, 0
    endif

    oShape->GetProperty, $
        ENTITY_TYPE=entityType, $
        N_ATTRIBUTES=nAttr, $
        N_ENTITIES=nEntity

    if (nEntity eq 0) then begin
        OBJ_DESTROY, oShape
        return, 0
    endif

    case entityType of
    1: idlshapetype = 'IDLSHAPEPOINT'
    3: idlshapetype = 'IDLSHAPEPOLYLINE'
    5: idlshapetype = 'IDLSHAPEPOLYGON'
    11: idlshapetype = 'IDLSHAPEPOINT'
    else: begin
        self->SignalError, $
            [unableRead, $
            'Cannot read shapefile entities of type: ' + $
                STRTRIM(entityType, 2)], $
            SEVERITY=2
        OBJ_DESTROY, oShape
        return, 0
    end
    endcase


    ; Two or three dimensional data.
    switch entityType of
    1: ; Point - fall thru
    3: ; Polyline - fall thru
    5: begin  ; Polygon
        ndim = 2
        break
       end
    11: ; PointZ - fall thru
    13: ; PolylineZ - fall thru
    15: begin  ; PolygonZ
        ndim = 3
        break
        end
    endswitch

    ; Minimum # of vertices. Polylines need 2, polygons need 3.
    minVert = (entityType eq 3) ? 2 : 3

    ; Best guess for name attribute.
    nameIndex = self._attributeIndex < nAttr

    filename = FILE_BASENAME(strFilename, '.shp', /FOLD_CASE)

    index = 0L

    if (self._combineAll) then begin
      vertexList = LIST()
      connList = LIST()
      nconn = 0LL
    endif else begin
      oData = OBJARR(nEntity)
      if (nameIndex ge 1) then $
        names = STRARR(nEntity)
    endelse

    ; Honor the LIMIT property if they aren't all equal (say to 0),
    ; and if they actually limit the globe to some extent.
    ; Otherwise it isn't worth the effort.
    isEqual = ARRAY_EQUAL(self._limit, self._limit[0])
    diffLon = ABS(self._limit[3] - self._limit[1])
    diffLat = ABS(self._limit[2] - self._limit[0])
    useLimit = ~isEqual && (diffLon lt 350 || diffLat lt 170)
    limit = self._limit

    entities = oShape->GetEntity(/ALL)

    for i=0,nEntity-1 do begin

        entity = entities[i]

        ; Remove tiny useless shapes.
        if (entityType ne 1 && ptr_valid(entity.vertices) && $
          n_elements(*entity.vertices) le 10) then continue

        if (useLimit) then begin
          xmin = entity.bounds[0]
          ymin = entity.bounds[1]
          xmax = entity.bounds[4]
          ymax = entity.bounds[5]
          ; If the latitude is out of range, then skip this entity
          if ((ymin lt limit[0] && ymax lt limit[0]) || $
            (ymin gt limit[2] && ymax gt limit[2])) then continue
          ; If the longitude is out of range, then shift over by 360 degrees
          if ((xmin lt limit[1] && xmax lt limit[1]) || $
            (xmin gt limit[3] && xmax gt limit[3])) then begin
            xmin += 360
            xmax += 360
            ; If the longitude is still out of range, then skip this entity
            if ((xmin lt limit[1] && xmax lt limit[1]) || $
              (xmin gt limit[3] && xmax gt limit[3])) then continue
          endif
        endif


        if (nAttr gt 0) then $
            attr = oShape->GetAttributes(i)

        isAntarctica = entity.n_vertices gt 1000 && entity.bounds[5] lt -55

        ; Create empty parameter set with chosen name.
        if (~self._combineAll) then begin
            name = (nameIndex ge 1) ? $
                STRMID(STRTRIM(attr.(nameIndex - 1), 2),0,40) : $
                (filename + ' ' + STRTRIM(i,2))
            name = STRJOIN(STRSPLIT(name, '/', /EXTRACT), '_')
            if (nameIndex ge 1) then $
                names[i] = name
            oData[i] = OBJ_NEW('IDLitParameterSet', $
                NAME=name, $
                ICON='drawing', $
                TYPE=idlshapetype, $
                DESCRIPTION=strFilename)
        endif


        switch entityType of

        11: ; PointZ, fall thru
        1: begin  ; Points
            point = entity.bounds[0:ndim-1]
            if (self._combineAll) then begin
                index++
                vertexList.Add, point
            endif else begin
                oVert = OBJ_NEW('IDLitData', point, $
                    NAME='Vertices', TYPE=idlshapetype, ICON='segpoly')
                oData[i]->Add, oVert, PARAMETER_NAME='Vertices'
            endelse
           end

        3: ; Polyline, fall thru
        5: begin  ; Polygon
            if (entity.n_vertices lt minVert) then $
                break

            if (~self._combineAll && N_ELEMENTS(connectivity)) then $
                void = TEMPORARY(connectivity)

            if (self._combineAll) then begin
                ; The SHAPES parameter is a vector, each element of which
                ; is the starting index within the CONNECTIVITY of the
                ; next shape. This allows multiple shapes to be stored
                ; within a single parameter set, but still have the
                ; IDLitVisPolygon tessellate them separately.
                shapes = (N_ELEMENTS(shapes) gt 0) ? $
                    [TEMPORARY(shapes), nconn] : nconn
            endif

            ; Number of polygons within the shape.
            if (entity.n_parts gt 1) then begin
                ; Construct a connectivity array.
                parts = [*entity.parts, entity.n_vertices]
                for part=0L, entity.n_parts - 1 do begin
                    pStart = parts[part]
                    nSubVert = parts[part + 1] - pStart
                    if (nSubVert lt minVert) then $
                        continue
                    if (isAntarctica) then begin
                        conn1 = self->_FixAntarctica(entity.vertices, $
                            pStart, nSubVert, index)
                    endif else begin
                        conn1 = [nSubVert, LINDGEN(nSubVert) + pStart + index]
                    endelse
                    if (self._combineAll) then begin
                      connList.Add, conn1
                      nconn += N_ELEMENTS(conn1)
                    endif else begin
                      connectivity = (N_ELEMENTS(connectivity) gt 0) ? $
                        [TEMPORARY(connectivity), conn1] : conn1
                    endelse
                endfor
            endif else begin
              if (isAntarctica) then begin
                conn1 = self->_FixAntarctica(entity.vertices, $
                  0, entity.n_vertices, index)
              endif else begin
                conn1 = [entity.n_vertices, LINDGEN(entity.n_vertices) + index]
              endelse
              if (self._combineAll) then begin
                connList.Add, conn1
                nconn += N_ELEMENTS(conn1)
              endif else begin
                connectivity = (N_ELEMENTS(connectivity) gt 0) ? $
                  [TEMPORARY(connectivity), conn1] : conn1
              endelse
            endelse

            if (self._combineAll) then begin
                index += entity.n_vertices
                vertexList.Add, *entity.vertices
            endif else begin
                oVert = OBJ_NEW('IDLitData', *entity.vertices, $
                    NAME='Vertices', TYPE=idlshapetype, ICON='segpoly')
                oData[i]->Add, oVert, PARAMETER_NAME='Vertices'
                if (N_ELEMENTS(connectivity) gt 0) then begin
                    oConn = OBJ_NEW('IDLitData', connectivity, $
                        NAME='Connectivity', TYPE='IDLCONNECTIVITY', $
                        ICON='segpoly')
                    oData[i]->Add, oConn, PARAMETER_NAME='Connectivity'
                endif
            endelse

            break
           end

        endswitch

        ; Add attributes if present.
        if (~self._combineAll && N_TAGS(attr) gt 0) then begin
            oAttr = OBJ_NEW('IDLitData', attr, $
                NAME='Attributes', TYPE='IDLSHAPEATTRIBUTES', $
                ICON='binary')
            oData[i]->Add, oAttr, PARAMETER_NAME='Attributes'
        endif

        oShape->DestroyEntity, entity

    endfor

    OBJ_DESTROY, oShape


    if (self._combineAll) then begin

        if (N_ELEMENTS(vertexList) gt 0) then begin
            oData = OBJ_NEW('IDLitParameterSet', $
                NAME=filename, $
                ICON='drawing', $
                TYPE=idlshapetype, $
                DESCRIPTION=strFilename)
            vertices = vertexList.ToArray(DIMENSION=2, /NO_COPY)
            oVert = OBJ_NEW('IDLitData', vertices, $
                NAME='Vertices', TYPE=idlshapetype, ICON='segpoly')
            oData->Add, oVert, PARAMETER_NAME='Vertices'
            if (N_ELEMENTS(connList) gt 0) then begin
              connectivity = connList.ToArray(DIMENSION=1, /NO_COPY)
                oConn = OBJ_NEW('IDLitData', connectivity, $
                    NAME='Connectivity', TYPE='IDLCONNECTIVITY', $
                    ICON='segpoly')
                oData->Add, oConn, PARAMETER_NAME='Connectivity'
            endif
            if (N_ELEMENTS(shapes) gt 0) then begin
                oShapeData = OBJ_NEW('IDLitData', shapes, $
                    NAME='Shapes', TYPE='IDLSHAPES', $
                    ICON='segpoly')
                oData->Add, oShapeData, PARAMETER_NAME='Shapes'
            endif
        endif

    endif else begin

        ; If using an attribute for name, sort the data objects.
        if (nameIndex ge 1) then begin
            oData = oData[sort(strupcase(names))]
        endif

    endelse

    ; Check all the separate data objects.
    good = WHERE(OBJ_VALID(oData), ngood)
    if (ngood gt 0) then begin
        oData = oData[good]
        return, 1
    endif

    return, 0

end


;---------------------------------------------------------------------------
; IDLitReadShapefile::Isa
;
; Purpose:
;   Method that will return true if the given file is a shapefile.
;
; Paramter:
;   strFilename  - The file to check
;
function IDLitReadShapefile::Isa, strFilename

    compile_opt idl2, hidden

    oShape = OBJ_NEW('IDLffShape')

    ; CT: Is this a sufficient test?
    ; What if the .dbf file is missing?
    success = oShape->Open(strFilename)

    ; Assume we have no attributes.
    enumlist = '<Shape index>'

    if (success) then begin
        oShape->GetProperty, N_ATTRIBUTES=nAttr, N_ENTITIES=nEntity
        if (nAttr gt 0) then begin
            oShape->GetProperty, ATTRIBUTE_NAMES=attrNames
            enumlist = [enumlist, attrNames]
            ; Best guess for name attribute.
            ; Add 1 for our default value of Shape index.
            self._attributeIndex = self->_GetNameAttribute(oShape) + 1

            if (nEntity ge 1) then begin
                ; Add a sample value to the end of each attribute.
                attr = oShape->GetAttributes(0)
                for i=0,nAttr-1 do begin
                    str = STRTRIM(attr.(i), 2)
                    if (~str) then $
                        continue
                    if (STRLEN(str) gt 19) then $
                        str = STRMID(str, 0, 16) + '...'
                    enumlist[i + 1] += ' (' + str + ')'
                endfor
            endif
        endif
    endif

    self->SetPropertyAttribute, 'ATTRIBUTE_NAME', $
        ENUMLIST=enumlist

    OBJ_DESTROY, oShape

    return, success

end


;---------------------------------------------------------------------------
; Definition
;---------------------------------------------------------------------------
; IDLitReadShapefile__Define
;
; Purpose:
; Class definition for the IDLitReadShapefile class
;
pro IDLitReadShapefile__Define

    compile_opt idl2, hidden

    void = {IDLitReadShapefile, $
        inherits IDLitReader, $
        _combineAll: 0b, $
        _attributeIndex: 0, $
        _limit: DBLARR(4) $
        }
end
