;+
; idl-disable potential-undefined-var, potential-var-use-before-def
;-

;  Copyright (c)  NV5 Geospatial Solutions, Inc. All
;        rights reserved. Unauthorized reproduction is prohibited.

; ----------------------------------------------------------------------------
;  cli_progress
;  class to manage and print a progress bar.


; initialize:
;  Basically a .Init. However because this is a class and not a object
;  we want to avoid wordage that might be confusing.
;  This sets up the default values for the progress bar and provides a
;  one stop shop to set any modifications to the defaults you would want.
pro cli_progress::initialize, _extra = _extra
  compile_opt idl2, hidden, nosave, static
  COMMON ShareIdlProgressBar, _maximum, _title, $
    _width, _complete_char, _pattern, $
    _complete, _lastLength, _incomplete_char, $
    _bartype, _reverse, _isInitialized,$ 
    _autoFinish, _text, _percent, _Value, _ticker,$ 
    _remaining, _clock, _speed
    
  on_error, 2
  
  _isInitialized = boolean(1)
  
  _Value = 0; 
  
  _ticker = 0;

  ;  This signifies what type of progress bar we are using. If you want to 
  ;  add more types add a flag here and it should let you give yourself room.
  _barType = 0

  ;  Width of the bar in characters
  _width = 40

  ;  Marks the conclusion of a series and determines how many segments to fill in
  _maximum = 100d
  
  ;  Decides if we will autofinish ProgressBars
  _autoFinish = boolean(1)

  ;  Character used for "loaded" chunks
  _complete_char = '#'

  ;  Character used for "unloaded" chunks
  _incomplete_char = '-'

  ;  Pattern to be played before loading bar
  _pattern = [""]

  ;  Message to be displayed before loading bar
  _title = !NULL

  ; Variables related to "ping pong" bars

  ;  reverse keyword means that the load_charcter is supposed to be displayed in order
  ;  for example for the character ,o0 if the bar is headed; 
  ;  ----------->
  ; [....,o0..]
  ; <----------
  ; [...0o,...]
  _reverse = boolean(0)

  ; End Variables related to "ping pong" bars
  
  ;  Internal flag to say if the progress bar is finished drawing
  _complete = boolean(0)

  ;  Internal count to handle variable length progress bars
  _lastLength = 0
  
  _text = !null
  
  _percent = !true
  
  _autoFinish = 1
  
  _remaining = 0

  ;  Set anything the user wants to set
  cli_progress.SetProperty, _extra = _extra

end

;---------------------------------------------------------------------------------


; SetProperty:
; used to set one or many preferences at a time.

;  Any changes to progressbar should idealy be done with dot notation, e.g.:
;  cli_progress.maximum=200 or by using the initialize method.
pro cli_progress::SetProperty, MAXIMUM = maximum, TITLE = title, WIDTH = width, LAST_STEP = laststep,$
  COMPLETE_CHAR = complete_char, SPINNER = pattern, TEXT = textint, PERCENT = percent, PRESET = preset,$
  INCOMPLETE_CHAR = incomplete_char, PING_PONG = PING_PONG, REVERSE = reverse, AUTO_FINISH = auto_finish, REMAINING = remaining
  compile_opt idl2, hidden, nosave, static
  COMMON ShareIdlProgressBar
  on_error, 2
  ; Must be first. if Pingpong set some presets. Let the user overwrite them
  if isa(PING_PONG) then begin
    if fix(PING_PONG) eq boolean(0) then begin
      _barType = 0
      _percent = !True
      _maximum = 100
      _complete = !False
    endif else begin
      _barType = 1
      _percent = !false
      _maximum = !VALUES.F_INFINITY
      _complete = !False
    endelse
  endif
  
  if isa(preset) then begin
    if ~isa(preset,/string) then message,"PRESET must be a string."

    presetName = STRLOWCASE(strtrim(preset))
    switch presetName of
      "diamonds": begin
        cli_progress.complete_char = "◆"
        cli_progress.incomplete_char = "◇"
        break
      end
      "retro": begin
        cli_progress.complete_char = "="
        cli_progress.incomplete_char = "-"
        cli_progress.spinner = 1
        break
      end
      "ping_pong": begin
        cli_progress.Ping_Pong = !true
        cli_progress.complete_char = "<>"
        cli_progress.incomplete_char = "-"
        break
      end
      "shades": begin
        cli_progress.complete_char = '█'
        cli_progress.incomplete_char = "░"
        break
      end
      
      "squares": begin
        cli_progress.complete_char = '■'
        cli_progress.incomplete_char = " "
        break
      end
      else: begin
        ; Do nothing
      end
    endswitch
  endif
  
  if isa(maximum) then begin
    _maximum = FLOOR(maximum)
    if _maximum lt 0 then message,"MAXIMUM must be a positive number."
  endif
  
  if isa(title) then begin
      _title = `${title[0]}`
  endif
  
  if isa(pattern) then begin
    switch pattern of
      0: begin
        _pattern = [""]
        break 
        end
      1: begin
        _pattern = ["|","/","-","\"]
        break 
        end
      2: begin
        _pattern = ["◴","◷","◶","◵"]
        break 
        end
      3: begin
        _pattern = ["▁","▃","▄","▅","▆","▇","█","▇","▆","▅","▄","▃"]
        break 
        end
      4: begin
        _pattern = ["⠋", "⠙", "⠹", "⠸", "⠼", "⠴", "⠦", "⠧", "⠇", "⠏"]
        break
      end
      else: begin
        ; Do nothing
      end
    endswitch

  endif
  
  if isa(width) then begin
    _width = fix(width)
    if _width lt 0 then message,"WIDTH must be a positive integer."
    _width = width
  endif
  
  if isa(complete_char) then begin
    if ~isa(complete_char,/string) then message, "COMPLETE_CHAR must be a string."
    _complete_char = `${complete_char[0]}`
  endif
  
  if isa(incomplete_char) then begin
    if ~isa(incomplete_char,/string) then  message, "INCOMPLETE_CHAR must be a string."
    _incomplete_char = `${incomplete_char[0]}`
  endif
  
  if isa(reverse) then begin
    _reverse = fix(reverse,TYPE = 2)
  endif
  
  if isa(auto_finish) then begin
    _autoFinish = fix(auto_finish,TYPE = 2)
  endif
  
  if isa(textint) then begin
    _text = `${textint[0]}`
  endif
  
  if isa(percent) then begin
    _percent = keyword_set(percent)
  endif
  
  if isa(remaining) then begin
    _remaining = keyword_set(remaining)
    _speed = 0d
    _clock = 0d
  endif
end


;---------------------------------------------------------------------------------
;Helper functions for the making of loading bars.

; This function calculates and decides how do display the percent keyword.
; It returns a string tha tis the percent in its formated form without the %
function CLI_PROGRESS_CalcPercent, step
COMMON ShareIdlProgressBar

percent = step / _maximum * 100d
CASE !TRUE OF
  (_maximum gt 10000): StringToAdd = `${percent, "%7.3f"}% `
  (_maximum gt 1000):  StringToAdd = `${percent, "%6.2f"}% `
  (_maximum gt 100):   StringToAdd = `${percent, "%5.1f"}% `
  ELSE: StringToAdd = `${percent, "%3d"}% `
ENDCASE

return, StringToAdd
end

;  take care of cleaning larger loading bars.
;  if the previsous loading bar looked something like
;  I'm a long title :[...]
;  and then the second loading bar looked like
; [...]
; without this step the print would look like
; [...] long title :[...]
;  so we print spaces to cover up the extra trash.
; This function returns a string. That is the number os spaces required to clean any cruft from the old bar.
; Its original intention was to be added to the end of the bar.
function CLI_PROGRESS_Filler, currentLength, modifiedPattern
COMMON ShareIdlProgressBar
  modification = ""
  ; take into account multibyte loading patterns
  if isa(modifiedPattern) then begin
    if strlen(modifiedPattern) gt 1 then currentLength -= strlen(modifiedPattern) + 1
  endif
  
  if currentLength lt _lastLength then begin
    filler = (' ').dup(strlen(_complete_char))
    modification = STRING((filler).dup(_lastLength - currentLength))
  endif
  
  return, modification
end

; This function calculates the remaining keyword and returns a string with the proper information in the proper format.
; This string is then added to the progress bar.
; This function returns a string in the format of "<time> `remaining`" to be added to the loading bar.
function CLI_PROGRESS_DoREMAINING, step
  COMMON ShareIdlProgressBar
  ; if the bar is going to be completed. Don't print remaining. This is to give the bar a feeling of completedness.
  if step eq _maximum then return, ""
  
  ; if its the first step with REMAINING then clock is not set yet.
  ; start the timer.
  if _clock eq 0d then begin
    _clock = SYSTIME(1)
    return, ""
  endif
  
  ; Get a time since last update. This gives us a Speed per update.
  time_since_last = SYSTIME(1) - _clock
  _clock = SYSTIME(1)
  
  ; Init speed for first real step.
  if _speed eq 0 then _speed = time_since_last
  
  ; change weight based on num loops. This is a non-standard equation to try and stabilize
  ; large loops. It is not necissary and I originally weighted it to .1
  alpha = 100d/(1000d + sqrt(_maximum))
  
  ; Calc time remaining.
  _speed = (_speed * (1 - alpha)) + (time_since_last * alpha)
  timeremaining = _speed * (_maximum - step)
  
  ; Get times in human readable form.
  hour = floor(timeremaining/3600d)
  minute = floor((timeremaining mod 3600d)/60)
  second = floor(timeremaining mod 60d)
  
  ; Format said times.
  case (!TRUE) of
    (hour ne 0): estimate = `${hour}h:${minute}m`
    (minute ne 0): estimate = `${minute}m:${second}s`
    else: estimate = `${second lt 1 ? '<' : ''}${second > 1}s`
  endcase

  return, ` ${estimate} Remaining`
end


;---------------------------------------------------------------------------------
; End helper functions for making loading bars.

; This is a internal function.
; Handles the creation of the string that will make up
; a normal progress bar.
function cli_progress::_MakeProgressBarString, Step
  compile_opt idl2, hidden, nosave, static
  COMMON ShareIdlProgressBar
  on_error, 2
  
  n = n_elements(_pattern)
  nstars = fix(step / double(_maximum) * double(_width))
  
  ; Assemble basic loading bar.
  loadingBar = STRING((_complete_char).dup(nstars) + (_incomplete_char).dup(_width - nstars))
  
  ; Assemble loading pattern.
  rtrnstr = ""
  
  ; Add pattern if pattern exists.
  if fix(total(strlen(_pattern))) gt 0 then begin
    modifiedPattern = _pattern[_ticker mod n]
    rtrnstr += `[${modifiedPattern}] `
    _ticker += 1
  endif
  
  ; Add title if title exists.
  if (isa(_title) && _title ne '') then begin
    if strlen(`${_title}`) gt 0 then rtrnstr = rtrnstr+`${_title} `
  endif
  
  ; Add percent provided percent is not set to false.
  if _percent then begin
    rtrnstr += CLI_PROGRESS_CalcPercent(step)
  endif
  
  ; Add Loading bar. Anything after this will be after the body of the loading bar.
  rtrnstr = `${rtrnstr}[${loadingBar}]`

  if _remaining then begin
    rtrnstr += `${CLI_PROGRESS_DoREMAINING(step)}`
  endif
  
  ; Finally add any text the user might want to add.
  if (isa(_text) && _text ne '') then begin
    rtrnstr += ` ${_text}`
  endif
  
  ; Do we need filler? if so fill.
  currentLength = strlen(rtrnstr)
  rtrnstr += CLI_PROGRESS_Filler(currentLength, modifiedPattern)
  _lastLength = currentLength
  
  return, rtrnstr
end

;---------------------------------------------------------------------------------

; This is a internal function.
; Handles the creation of the string that will make up
; a "ping pong" progress bar.
function cli_progress::_MakePING_PONGBarString, Step
  compile_opt idl2, hidden, nosave, static
  COMMON ShareIdlProgressBar
  on_error, 2
  
  ; Prep contents of the loading bar.
  section = fix(Step/_width) + 1
  n = n_elements(_pattern)
  
  ; BarStep will be the index where we are in the loading bar.
  ; So if the width is 30 it will count 0-30.
  barStep = Step mod _width
  
  ; We want to move to the right on odd and to the left on even.
  odd = section mod 2
  if odd then begin
    loadingBar = STRING(_incomplete_char.dup(barStep) $
      + _complete_char + _incomplete_char.dup(_width - barStep))
  
  ; else (even) move left  
  endif else begin
    modLoadCharacter = _complete_char
    ; handle _reverse keyword
    if _reverse eq boolean(1) then begin
      modLoadCharacter = modLoadCharacter.reverse()
    endif
    loadingBar = STRING(_incomplete_char.dup(_width - barStep) $
      + modLoadCharacter + _incomplete_char.dup(barStep))
  endelse
  
  ; Assemble loading pattern.
  rtrnstr = ""
  ; Add pattern if pattern keyword exists.
  if fix(total(strlen(_pattern))) gt 0 then begin
    modifiedPattern = _pattern[_ticker mod n]
    rtrnstr += `[${modifiedPattern}] `
    _ticker += 1
  endif
  
  ; Add title if title exists.
  if (isa(_title) && _title ne '') then begin
    rtrnstr += `${_title} `
  endif
  
  ; Add percent provided percent is not set to false.
  if _percent AND _maximum ne !VALUES.F_INFINITY then begin
    rtrnstr +=  CLI_PROGRESS_CalcPercent(step)
  endif
  
  ; Add the Loading bar.
  rtrnstr = `${rtrnstr}[${loadingBar}]`
  
  ; add remaining if remaining keyword exists.
  if _remaining and _maximum ne !VALUES.F_INFINITY then begin
    rtrnstr += `${CLI_PROGRESS_DoREMAINING(step)}`
  endif

  ; finally add any text the user might want to add.
  if (isa(_text) && _text ne '') then begin
    rtrnstr += ` ${_text}`
  endif
  
  ; do we need filler? if so fill
  currentLength = strlen(rtrnstr)
  rtrnstr += CLI_PROGRESS_Filler(currentLength, modifiedPattern)
  _lastLength = currentLength

  return, rtrnstr
end

;---------------------------------------------------------------------------------

; This is a internal method.
; Checks to make sure a progress bar is initialized and if its not we go ahead and initialize it.
; We should be doing this before we do anything that might use an internal variable.
pro cli_progress::_quickCheck
  compile_opt idl2, hidden, nosave, static
  COMMON ShareIdlProgressBar
  on_error, 2
  ;  if the user trys to do something weird like run .update before .initialize. Just run a .initialize for them.
  if ~isa(_isInitialized) then begin
   cli_progress.initialize
  endif
end

;---------------------------------------------------------------------------------

; Update:
; Updates the progress bar.
; The creation of the progress bar string to the corresponding bar type.
pro cli_progress::Update, step, TEXT = text, OUTPUT = output
  compile_opt idl2, hidden, nosave, static
  COMMON ShareIdlProgressBar
  on_error, 2
  ; Check to make sure we are initialized.
  cli_progress._quickCheck

  if isa(text) then cli_progress.SetProperty, TEXT = text

  _value = isa(step) ? step : _value + 1
  
  intStep = _value
  
  ; dont print the bar if the bar is finished.
  if intStep le _maximum then begin
    
    if _autoFinish eq !TRUE then begin
      willBeCompleted = (_value eq _maximum)
      if _complete eq !TRUE and willBeCompleted eq !TRUE then return
      _complete = willBeCompleted
    endif

    switch _barType of
      0: begin
        str = cli_progress._MAKEPROGRESSBARSTRING(double(intStep))
        break
      end
      1: begin
        str = cli_progress._MAKEPING_PONGBARSTRING(double(intStep))
        break
      end
      else: begin
        message, "non-Valid bar type."
      end
    endswitch
    
    wePrint = ~arg_present(output)
    if wePrint then begin
      print, `\r${str}`, newline = _complete
    endif else begin
      output = `${str}`
      _lastLength = 0
    endelse
  endif
  ; if we have completed then we have carrage returned.(/n) then we dont have any last length.
  if _complete then _LastLength = 0
end

;---------------------------------------------------------------------------------

; newline:
; Method to release controls, This only passes a newline in.
; Its use is to "break" the progress bar. and return control of the
; STDOUT to the user.
pro cli_progress::newline
  compile_opt idl2, hidden, nosave, static
  on_error, 2
  print, ""
end

pro cli_progress::erase
  compile_opt idl2, hidden, nosave, static
  COMMON ShareIdlProgressBar
  on_error, 2
  
  buffer = _lastLength
  if fix(total(strlen(_pattern))) gt 0 then begin
    n = n_elements(_pattern)
    modifiedPattern = _pattern[_ticker mod n]
  endif
   
  if isa(modifiedPattern) then begin
    buffer += strlen(modifiedPattern) + 1
  endif
  
  print, `\r${(" ").dup(buffer)}\r`, newline = 0
end

;---------------------------------------------------------------------------------


; Finish:
; Finishes a progress bar no matter the progess.
; If the _autoFinish keyword is set. Then this and Newline will be the only way
; to stop the progress bar.
pro cli_progress::Finish, OUTPUT = output, TEXT = text
  compile_opt idl2, hidden, nosave, static
  COMMON ShareIdlProgressBar
  on_error, 2
  ;  check to make sure we are initialized
  cli_progress._quickCheck
  
  ; prep OUTPUT keyword
  weprint = ~arg_present(output)
  
  if (_complete) eq !TRUE then return
    
    ;Prep a completing package.
    ;Complete = true.
    ;Dont try and hold for a autoFinish as we are finishing now.
    autoFinishStore = _autoFinish
    _autoFinish = !FALSE
    _complete = boolean(1)
  
    switch _barType of
      0: begin
        if wePrint then begin
          cli_progress.update, _maximum, TEXT = text
        endif else begin
          cli_progress.update, _maximum, OUTPUT = output, TEXT = text
        endelse
        break
      end
      1: begin
        step = min([_value,_maximum])
        if wePrint then begin
          cli_progress.update, step, TEXT = text
        endif else begin
          cli_progress.update, step, OUTPUT = output, TEXT = text
        endelse
        break
      end
      else: begin
        message, "non-Valid bar type."
      end
    endswitch
    _autoFinish = autoFinishStore
end
;---------------------------------------------------------------------------------

pro cli_progress__define
compile_opt idl2, hidden, nosave
  !null = {cli_progress, inherits IDL_Object}
end
