

;+
; :Description:
;   The FLOAT16_DECODE function converts a blob of 16-bit "half-precision"
;   floating-point data values into 32-bit single-precision floating point numbers.
;   The function assumes that the 16-bit input values are stored using the
;   IEEE 754 binary16 (or float16) format, where the bits are "seeeeeffffffffff",
;   where "s" is the sign bit, "eeeee" is the 5-bit exponent (with an offset of 15),
;   and the f's are 10 bits for the fractional part. The significand actually
;   has 11 bits of precision because it has an implicit lead bit of 1
;   (unless the exponent bits are all zero).
;
;   Because IDL does not have a native 16-bit float data type, the input data
;   must be of type Byte, Int, or Uint. For Byte input, it is assumed that each
;   pair of bytes represents a single 16-bit binary16 number. For Int or Uint,
;   it is assumed that each value represents the bits of the 16-bit binary16
;   number. In all cases, the bits will be decoded using the binary16 format,
;   and then converted to 32-bit single-precision floats.
;   
;   Note that binary16 numbers have a limited range and precision, and are
;   only suitable for certain applications. The smallest nonzero number
;   is +/-5.96e-08, while the largest is +/-65504. Numbers larger than
;   this are set to Infinity.
;
; :Returns: A scalar or array of 32-bit Float numbers. If the input is an array,
;   then the result will be an array of the same dimensions.
;
; :Arguments:
;   array: in, required, Byte | Int | Uint
;     Set this argument to a scalar or array of type Byte, Int, or Uint.
;     If this argument is type Byte, then each pair of bytes will be combined
;     into a single Uint number before converting to a 32-bit Float.
;
;-
function float16_decode, array
  compile_opt idl2
  on_error, 2

  if (~isa(array, 'byte') && ~isa(array, 'int') && ~isa(array, 'uint')) then begin
    message, 'Input must be type BYTE, INT, or UINT.'
  endif

  x = isa(array, 'byte') ? uint(array, 0, n_elements(array) / 2) : uint(array)
  ; Convert a two-byte input to a scalar output
  if (isa(array, 'byte') && n_elements(array) eq 2) then begin
    x = x[0]
  endif

  ; First find any +/- infinity values
  infpos = where(x eq 0x7c00, /null)
  infneg = where(x eq 0xfc00, /null)

  ; Pull out the leading sign bit, convert to +/-1
  sign = 1.0 - (x and 32768u) / 16384u

  ; NaNs are at x11111xxxxxxxxxx, i.e. where the exponent = 31
  exponent = ishft(x, -10) and 31u
  nans = where(exponent eq 31, /null)

  offset = exponent ne 0
  exponent = 2.0^(exponent - offset - 14.0)
  significand = x and 1023
  fraction = offset + significand / 1024.0
  result = exponent * fraction * sign

  ; Fill in NaNs before infinity, since they both have the same exponent.
  result[nans] = !values.f_nan
  result[infpos] = !values.f_infinity
  result[infneg] = -!values.f_infinity
  return, result
end
