; Copyright (c)  NV5 Geospatial Solutions, Inc. All
; rights reserved. Unauthorized reproduction is prohibited.
;
;+
; CLASS_NAME:
;    ASDF_NDArray
;
; PURPOSE:
;    The ASDF_NDArray class is used to create an ASDF ndarray object,
;    which is a subclass of YAML_Map.
;
; CATEGORY:
;    Datatypes
;
;-

; ---------------------------------------------------------------------------
function ASDF_NDArray::Init, data, _EXTRA=ex
  compile_opt idl2, hidden
  on_error, 2
  !null = self.Dictionary::Init()
  self.compression = 'none'
  self.ASDF_NDArray::SetProperty, data=data, tag='!core/ndarray-1.0.0', _EXTRA=ex
  if (~isa(data)) then self.SetData
  return, 1
end

; ---------------------------------------------------------------------------
function ASDF_NDArray::ToBytes
  compile_opt idl2, hidden
  on_error, 2

  data = self.GetData()
  nbytesPerElement = ([0,1,2,4,4,8,8,0,0,16,0,0,2,4,8,8])[data.type]
  n = data.length
  bytedata = fix(temporary(data), 0, n * nbytesPerElement, type=1)
  return, bytedata
end

; ---------------------------------------------------------------------------
; Convert an ASDF_NDArray to a binary data block structure
;
function ASDF_NDArray::GetBlock
  compile_opt idl2, hidden
  bytedata = self.toBytes()
  data_size = swap_endian(ulong64(bytedata.length), /swap_if_little_endian)
  compression = self.compression
  case compression of
    '': compression = bytarr(4)
    'none': compression = bytarr(4)
    else: begin
      if (compression ne 'zlib') then begin
        message, `Unsupported compression '${compression}'. Defaulting to zlib.`, /informational
      endif
      compression = byte('zlib')
      bytedata = zlib_compress(bytedata)
    end
  endcase
  used_size = swap_endian(ulong64(bytedata.length), /swap_if_little_endian)
  block = { $
    magic: byte([0xd3, 0x42, 0x4c, 0x4b]), $
    header_size: swap_endian(48us, /swap_if_little_endian), $
    flags: swap_endian(0ul, /swap_if_little_endian), $
    compression: compression, $
    alloc_size: used_size, $
    used_size: used_size, $
    data_size: data_size, $
    checksum: bytarr(16), $   ; TODO: calculate checksum
    data: bytedata $
  }
  return, block
end

; ---------------------------------------------------------------------------
function ASDF_NDArray::ConvertToIDL, data
  compile_opt idl2, hidden
  on_error, 2

  obj = self
  datatype = obj.datatype

  ; Map ASDF datatypes to IDL types - index gives the IDL type number
  asdfToIDL = ['', 'uint8', 'int16', 'int32', 'float32', 'float64', $
    'complex64', '', '', 'complex128', '', '', 'uint16', 'uint32', 'int64', 'uint64']
  idltype = (where(asdfToIDL eq datatype))[0]
  if (idltype le 0) then begin
    case datatype of
      'int8': idltype = 1
      'bool8': idltype = 1
      else: message, `Unsupported datatype: '${datatype}'`
    endcase
  endif

  nbytesPerElement = ([0,1,2,4,4,8,8,0,0,16,0,0,2,4,8,8])[idltype]
  n = data.length
  data = fix(temporary(data), 0, n / nbytesPerElement, type=idltype)

  if (obj.hasKey('strides')) then begin
    strides = obj.strides / nbytesPerElement
    offset = obj.hasKey('offset') ? (obj.offset / nbytesPerElement) : 0
    message, 'Reading data from strides is not supported.'
    print, strides
    print, offset
  endif

  if (obj.hasKey('byteorder')) then begin
    case obj.byteorder of
      'little': swap_endian_inplace, data, /swap_if_big_endian
      'big': swap_endian_inplace, data, /swap_if_little_endian
      else:
    endcase
    ; Now reset our flag to match our platform.
    isLittle = swap_endian(1L, /swap_if_little_endian) ne 1L
    obj.byteorder = isLittle ? 'little' : 'big'
  endif

  if (obj.hasKey('shape')) then begin
    data = reform(data, reverse(obj.shape), /overwrite)
  endif

  if (datatype eq 'bool8') then begin
    data = boolean(temporary(data))
  endif
  return, data
end

; ---------------------------------------------------------------------------
function ASDF_NDArray::GetData
  compile_opt idl2, hidden
  on_error, 2

  ; Is the data embedded within the ASDF file?
  if (self.hasKey('data')) then begin
    self.Dictionary::GetProperty, data = data
    return, data
  endif

  ; Need to read data from an external file and cache it.
  lun = 0
  catch, err
  if (err ne 0) then begin
    catch, /cancel
    if (lun ne 0) then free_lun, lun, /force
    msg = `Unable to read data from file${isa(self._filename) ? `: '${self._filename}'` : '.'}`
    message, msg
  endif

  openr, lun, self._filename, /get_lun
  point_lun, lun, self._datastart
  data = bytarr(self._datasize, /nozero)
  readu, lun, data
  free_lun, lun, /force
  catch, /cancel

  case (self.compression) of
    '':
    'none':
    'zlib': data = zlib_uncompress(data)
    else: message, `Compression not supported: '${self.compression}'`
  endcase

  data = self.ConvertToIDL(data)
  obj = self
  ; Set using brackets so our key is lowercase
  obj['data'] = data
  return, data
end

; ---------------------------------------------------------------------------
pro ASDF_NDArray::SetData, data
  compile_opt idl2, hidden
  on_error, 2

  obj = self
  ; Set all of these using brackets so our keys are lowercase
  if (isa(data)) then begin
    obj['shape'] = isa(data,/scalar) ? [1] : reverse(size(data, /dimensions))
    ; Reset our flag to match our platform.
    isLittle = swap_endian(1L, /swap_if_little_endian) ne 1L
    obj['byteorder'] = isLittle ? 'little' : 'big'
    datamap = ['', 'uint8', 'int16', 'int32', 'float32', 'float64', 'complex64', $
      '', '', 'complex128', '', '', 'uint16', 'uint32', 'int64', 'uint64']
    obj['datatype'] = datamap[data.type]
    obj['data'] = isa(data,/scalar) ? [data] : data
    if (~obj.hasKey('source') && data.length gt 10) then begin
      obj['source'] = 0
    endif
  endif else begin
    obj['shape'] = [0]
    obj['byteorder'] = 'little'
    obj['datatype'] = 'float64'
  endelse
end

; ---------------------------------------------------------------------------
function ASDF_NDArray::_overloadForeach, value, key
  compile_opt idl2, hidden
  on_error, 2
  result = self.Dictionary::_overloadForeach(value, key)
  ; If we have a 'source' then skip the data field
  if (key.toLower() eq 'data') then begin
    if (self.hasKey('source')) then begin
      result = self.Dictionary::_overloadForeach(value, key)
    endif else begin
      ; Normally, yaml_serialize would write out a byte array as !!binary.
      ; Instead, return an integer array to fool it into writing out numbers.
      ; This should not affect users getting the data via ['data'] or .data
      ; because that will go through ::GetData up above and return a bytarr.
      if (isa(value, 'byte') && ~isa(value, /boolean)) then begin
        value = fix(value)
      endif
    endelse
  endif
  return, result
end

; ---------------------------------------------------------------------------
function ASDF_NDArray::Get, key
  compile_opt idl2, hidden
  on_error, 2

  if (key.toLower() eq 'data') then begin
    return, self->GetData()
  endif
  result = self.Dictionary::Get(key)
  return, result
end

; ---------------------------------------------------------------------------
pro ASDF_NDArray::GetProperty, $
  alias=alias, anchor=anchor, tag=tag, value=value, $
  storage=storage, data=data, $
  _datastart=_datastart, _datasize=_datasize, _filename=_filename, $
  compression=compression, _ref_extra=ex
  compile_opt idl2, hidden
  on_error, 2

  ; Need to handle all of these so they don't become keys in the dictionary.
  self.YAML_Node::GetProperty, alias=alias, anchor=anchor, tag=tag, value=value
  if arg_present(_filename) then _filename = self._filename
  if arg_present(_datastart) then _datastart = self._datastart
  if arg_present(_datasize) then _datasize = self._datasize
  if arg_present(compression) then compression = self.compression
  if arg_present(data) then begin
    data = self->GetData()
  endif

  if (arg_present(storage)) then begin
    obj = self
    source = obj.hasKey('source') ? obj['source'] : !null
    storage = 'inline'
    if (isa(source, /number)) then begin
      storage = 'internal'
    endif else if (isa(source, /string)) then begin
      storage = 'external'
    endif
  endif

  if (isa(ex)) then begin
    self.Dictionary::GetProperty, _extra=ex
  endif
end

; ---------------------------------------------------------------------------
pro ASDF_NDArray::SetProperty, $
  alias=alias, anchor=anchor, tag=tag, value=value, $
  storage=storage, data=data, source=source, $
  _datastart=_datastart, _datasize=_datasize, _filename=_filename, $
  compression=compression, _extra=ex
  compile_opt idl2, hidden
  on_error, 2

  ; Need to handle all of these so they don't become keys in the dictionary.
  self.YAML_Node::SetProperty, alias=alias, anchor=anchor, tag=tag, value=value

  if (isa(storage)) then begin
    obj = self
    source = obj.hasKey('source') ? obj['source'] : !null
    case storage.toLower() of
      'inline': if (source) then obj.Remove, 'source'
      'internal': if (~isa(source, /number)) then obj['source'] = 0
      'external': if (~isa(source, /string)) then obj['source'] = ''
    endcase
  endif else if (isa(source)) then begin
    obj = self
    obj['source'] = source
  endif

  if (isa(data)) then begin
    self->SetData, data
  endif

  if (isa(_filename)) then self._filename = _filename
  if (isa(_datastart)) then self._datastart = _datastart
  if (isa(_datasize)) then self._datasize = _datasize
  if (isa(compression)) then self.compression = compression
  if (isa(ex)) then begin
    self.Dictionary::SetProperty, _extra=ex
  endif
end

; ---------------------------------------------------------------------------
function ASDF_NDArray::_overloadHelp, varname
  compile_opt idl2, hidden
  result = varname + ('               ').substring(varname.strlen()) + $
  ` ${obj_class(self)}  <ID=${obj_valid(self, /get_heap_identifier)}>`
  extra = ''
  if (self.hasKey('datatype')) then extra += ` ${self['datatype']}`
  if (self.hasKey('shape')) then begin
    shape = self['shape']
    extra += shape.length gt 1 ? ` ${shape}` : ` [${shape}]`
  endif
  if (self.hasKey('source')) then begin
    extra += isa(self['source'], /number) ? ' internal' : ' external'
  endif else begin
    extra += ' inline'
  endelse
  if (extra) then result += ' ' + extra
  return, result
end

; ---------------------------------------------------------------------------
pro ASDF_NDArray__DEFINE
  compile_opt idl2, hidden
  void = {ASDF_NDArray, $
    inherits YAML_Node, $
    inherits Dictionary, $
    _filename: '', $
    _datastart: 0LL, $
    _datasize: 0LL, $
    compression: ''}
end
