; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;----------------------------------------------------------------------------
; Purpose:
;   The IDLitVisLegendFillPlotItem class is the component wrapper
;   for the FillPlot item subcomponent of the legend.
;
;

;----------------------------------------------------------------------------
; Purpose:
;   Initialize this component
;
function IDLitVisLegendFillPlotItem::Init, _REF_EXTRA=_extra
  compile_opt idl2, hidden
  
  ; Initialize superclass
  if (~self->IDLitVisLegendItem::Init( $
      NAME="FillPlot Legend Item", $
      DESCRIPTION="A FillPlot Legend Entry", $
      _EXTRA=_extra)) then begin
    return, 0
  endif
  
  return, 1 ; Success
end


;----------------------------------------------------------------------------
; IDLitVisLegendFillPlotItem::Restore
;
; Purpose:
;   This procedure method performs any cleanup work required after
;   an object of this class has been restored from a save file to
;   ensure that its state is appropriate for the current revision.
;
pro IDLitVisLegendFillPlotItem::Restore
  compile_opt idl2, hidden
  
  ; Call superclass restore.
  self->IDLitVisLegendItem::Restore
  
  ; Call ::Restore on each aggregated ItVis object
  ; to ensure any new properties are registered.  Also
  ; call its UpdateComponentVersion method so that this
  ; will not be attempted later
  if (Obj_Valid(self._oItSymbol)) then begin
    self._oItSymbol->Restore
    self._oItSymbol->UpdateComponentVersion
  endif
end


;----------------------------------------------------------------------------
pro IDLitVisLegendFillPlotItem::RecomputeLayout, _REF_EXTRA=_extra
  compile_opt idl2, hidden
  
  oTool = self->GetTool()
  self->GetProperty, PARENT=oParent
  if (Obj_Valid(oTool) && Obj_Valid(oParent)) then begin
    oWindow = oTool->GetCurrentWindow()
    if (~Obj_Valid(oWindow)) then begin
      return
    endif
    textDimensions = oWindow->GetTextDimensions(self._oText)
    if (Obj_Valid(self._oPolygon)) then begin
      xdata = self._sampleWidth
      yLow = -0.4*textDimensions[1]
      yHigh = 0.4*textDimensions[1]
      ; Add an extra point in the middle in case there is a plot symbol.
      self._oPolygon->SetProperty, $
        DATA=[[0, yLow], [0, yHigh], [xdata, yHigh], [xdata, yLow], [0, yLow]] 
      self._oPolyline->SetProperty, $
        DATA=[[0, yLow], [0, yHigh], [xdata, yHigh], [xdata, yLow], [0, yLow]]
      self._oText->SetProperty, $
        LOCATIONS=[[self._sampleWidth+self._horizSpacing, 0]]
    endif
  endif
  
  self->UpdateSelectionVisual
  
  ; Update the upper level legend
  self->GetProperty, PARENT=oLegend
  if Obj_Valid(oLegend) then begin
    oLegend->RecomputeLayout
  endif
end


;----------------------------------------------------------------------------
pro IDLitVisLegendFillPlotItem::BuildItem
  compile_opt idl2, hidden
  
  ; Call our superclass first to set our properties.
  self->IDLitVisLegendItem::BuildItem
  
  self->AddOnNotifyObserver, self->GetFullIdentifier(), $
    self._oVisTarget->GetFullIdentifier()
    
  self._oVisTarget->GetProperty, $
    COLOR=color, $
    FILL_BACKGROUND=fillBack, $
    FILL_COLOR=fillColor, $
    LINESTYLE=linestyle, $
    PATTERN_BITMAP=patternBitmap, $
    PATTERN_ORIENTATION=patternOrientation, $
    PATTERN_SPACING=patternSpacing, $
    PATTERN_THICK=patternThick, $
    THICK=thick, $
    TRANSPARENCY=transparency, $
    NAME=name
    
  if (N_Elements(name) eq 0) then begin
    name=''
  endif

  oPattern = !NULL
  if (patternSpacing ne 0) then begin
    if (N_Elements(patternBitmap) le 1) then begin
      patternBitmap = !NULL
    endif
    oPattern = Obj_New('IDLgrPattern', $
                       ORIENTATION=patternOrientation, $
                       PATTERN=patternBitmap, $
                       SPACING=patternSpacing, $
                       STYLE=1+(N_Elements(patternBitmap) eq 128), $
                       THICK=patternThick)
  endif

  self._oPolygon = Obj_New('IDLgrPolygon', $
    ALPHA_CHANNEL=1-transparency/100., $
    COLOR=fillColor, $
    HIDE=~fillBack, $
    FILL_PATTERN=oPattern, $
    LINESTYLE=6, $
    /PRIVATE)
  self->Add, self._oPolygon
  self._oPolyline = Obj_New('IDLgrPolyline', $
    ALPHA_CHANNEL=1-transparency/100., $
    COLOR=color, $
    LINESTYLE=linestyle, $
    THICK=thick, $
    POLYLINES=[2,1,2,2,3,4], $
    /PRIVATE)
  self->Add, self._oPolyline
  
  self._oText->SetProperty, STRINGS=name
  
  self->RecomputeLayout
end


;----------------------------------------------------------------------------
; IIDLDataObserver Interface
;----------------------------------------------------------------------------
;;---------------------------------------------------------------------------
;; IDLitVisLegend::OnNotify
;;
;;
;;  strItem - The item being observed
;;
;;  strMessage - What happend. For properties this would be
;;               "SETPROPERTY"
;;
;;  strUser    - Message related data. For SETPROPERTY, this is the
;;               property that changed.
;;
;;
pro IDLitVisLegendFillPlotItem::OnNotify, strItem, StrMessage, strUser
  compile_opt idl2, hidden
  
  ; Let superclass handle other messages.
  if (strMessage ne 'SETPROPERTY') then begin
    ; Call our superclass.
    self->IDLitVisLegendItem::OnNotify, $
      strItem, StrMessage, strUser
    return
  endif
  
  oTool = self->GetTool()
  oSubject=oTool->GetByIdentifier(strItem)
  
  switch Strupcase(strUser) of
    'COLOR': begin
      oSubject->GetProperty, COLOR=color
      if (N_Elements(color) gt 0) then begin
        self._oPolyline->SetProperty, COLOR=color
      endif
      break
    end

    'FILL_BACKGROUND' : begin
      oSubject->GetProperty, FILL_BACKGROUND=fillBack
      self._oPolygon->SetProperty, HIDE=~fillBack
      break
    end

    'FILL_COLOR' : begin
      oSubject->GetProperty, FILL_COLOR=color
      if (N_Elements(color) gt 0) then begin
        self._oPolygon->SetProperty, COLOR=color
      endif
      break
    end
    
    'LINESTYLE': begin
      oSubject->GetProperty, LINESTYLE=linestyle
      if (N_Elements(linestyle) gt 0) then begin
        self._oPolyline->SetProperty, LINESTYLE=linestyle
      endif
      break
    end
    
    'THICK': begin
      oSubject->GetProperty, THICK=thick
      if (N_Elements(thick) gt 0) then begin
        self._oPolyline->SetProperty, THICK=thick
      endif
      break
    end
    
    'TRANSPARENCY': begin
      oSubject->GetProperty, TRANSPARENCY=transparency
      if (N_Elements(transparency) gt 0) then begin
        alpha = 1-transparency/100.
        self._oPolygon->SetProperty, ALPHA_CHANNEL=alpha
        self._oPolyline->SetProperty, ALPHA_CHANNEL=alpha
      endif
      break
    end
    
    'PATTERN_BITMAP' :
    'PATTERN_ORIENTATION' :
    'PATTERN_SPACING' :
    'PATTERN_THICK' : begin
      oSubject->GetProperty, PATTERN_BITMAP=patternBitmap, $
        PATTERN_ORIENTATION=patternOrientation, $
        PATTERN_SPACING=patternSpacing, $
        PATTERN_THICK=patternThick
      oPattern = Obj_New()
      if ( (patternSpacing ne 0) || $
           (N_Elements(patternBitmap) eq 128) ) then begin
        if (N_Elements(patternBitmap) le 1) then begin
          patternBitmap = !NULL
        endif
        oPattern = Obj_New('IDLgrPattern', $
          ORIENTATION=patternOrientation, $
          PATTERN=patternBitmap, $
          SPACING=patternSpacing, $
          STYLE=1+(N_Elements(patternBitmap) eq 128), $
          THICK=patternThick)
      endif
      self._oPolygon->SetProperty, FILL_PATTERN=oPattern
      break
    end

    else : ; ignore unknown parameters
  endswitch
end


;----------------------------------------------------------------------------
;+
; IDLitVisLegendFillPlotItem__Define
;
; PURPOSE:
;    Defines the object structure for an IDLitVisLegendFillPlotItem object.
;
;-
pro IDLitVisLegendFillPlotItem__Define
  compile_opt idl2, hidden
  
  struct = {IDLitVisLegendFillPlotItem, $
            inherits IDLitVisLegendItem, $
            _oPolygon: Obj_New(), $
            _oPolyline: Obj_New() $
           }
end
