; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;+
; :Description:
;   This compound widget displays a series of colors for the user to select one.  
;   It returns the selected color in an event structure via callback.
;
; :Category:
;   Compound widgets.
;
; :Params:
;   wCaller: The ID of the button associated with this widget. Typically, this is
;     a button with a color pick icon on it. The color picker will position itself
;     right below wCaller. If you make this icon part of a exclusive or nonexclusive
;     base it will highlight the button while the color picker is up (this is not
;     recommended in Unix, as it shows a selection button next to the icon).
;       
; :Keywords:; 
;   CALLBACK: The name of the routine to be called upon user selection of a color.
;     
; :Example:
;   ok = dialog_colorpicker(wButton, CALLBACK='myroutine')
;   ; when the user picks the 'red' color, there will be a call to 'myroutine'
;   ; with the parameter [255,0,0]
;-

;------------------------------------------------------------------------------
; dialog_colorpickerSpreadWidgets
;
; Purpose:
;   It is not possible to have several widgets in IDL on the same row base,
;   some aliged with the left edge and some aligned with the right edge. The
;   trick is to have an empty base in the middle with the exact size to be
;   able to position left and right sides appropriately.
;
pro dialog_colorpickerSpreadWidgets, wSeparator
  compile_opt idl2, hidden

  ; Get TLB
  next = wSeparator
  repeat begin
    wTLB = next
    next = Widget_Info(wTLB, /PARENT)
  endrep until (next eq 0)

  ; Account for window border/edge effect.
  borderSize = 25

  wBase = Widget_Info(wSeparator, /PARENT)
  wChildren = Widget_Info(wBase, /ALL_CHILDREN)
  if (N_Elements(wChildren) lt 3) then begin
    return
  endif

  xSizeAccumulated = 0.0
  foreach wChild, wChildren do begin
    if (wChild ne wSeparator) then begin
      geomChild = Widget_Info(wChild, /GEOMETRY)
      scrSize = Keyword_Set(vertical) ? geomChild.SCR_YSIZE : geomChild.SCR_XSIZE
      padding = Keyword_Set(vertical) ? geomChild.YPAD : geomChild.XPAD
      xSizeAccumulated += scrSize + padding
    endif
  endforeach

  geomTopBase = Widget_Info(wTLB, /GEOMETRY)
  Widget_Control, wSeparator, $
    SCR_XSIZE=(geomTopBase.SCR_XSIZE - $
    geomTopBase.XPAD*2 - $
    xSizeAccumulated - $
    borderSize) > 0

end

;------------------------------------------------------------------------------
; dialog_colorpickerHighlightBitmap
;
; Purpose:
;   Hightlights a color bitmap by drawing a black square around it.
;
pro dialog_colorpickerHighlightBitmap, array
  compile_opt idl2, hidden
  
  array[*,0,[0:2]] = 0
  array[*,-1,[0:2]] = 0
  array[0,*,[0:2]] = 0
  array[-1,*,[0:2]] = 0

end

;-----------------------------------------------------------------------------
; dialog_colorpickerGetAccumulatedWidgetOffset
;
function dialog_colorpickerGetAccumulatedWidgetOffset, wId, $
  OFFSET=offset, $
  TLB=tlb
  compile_opt idl2, hidden

  if (N_Elements(offset) eq 0) then begin
    offset = [0,0]
  endif

  if (~Widget_Info(wId, /VALID_ID)) then begin
    return, 1
  endif

  geometry = Widget_Info(wId, /GEOMETRY)
  offset[0] += geometry.XOFFSET
  offset[1] += geometry.YOFFSET
  if (~Widget_Info(Widget_Info(wId, /PARENT), /VALID_ID)) then begin
    ; We are at the top base, let's add the size of the window border.
    tlb = wId
    offset[0] += (geometry.SCR_XSIZE-geometry.XSIZE)
    offset[1] += (geometry.SCR_YSIZE-geometry.YSIZE)
  endif

  return, dialog_colorpickerGetAccumulatedWidgetOffset(Widget_Info(wId, /PARENT), $
    OFFSET=offset, TLB=tlb)
end

;----------------------------------------------------------------------------
;
function dialog_colorpickerValidateColors, colors, colorsOut
  compile_opt idl2, hidden

  colorsOut = colors

  if (ISA(colorsOut, 'list')) then colorsOut = colorsOut.toArray()
  if (N_ELEMENTS(colorsOut) eq 0) then return, 1
  if (~ISA(colorsOut, /NUMBER)) then return, 0
  if (MAX(colorsOut) gt 255) then return, 0
  if (MIN(colorsOut) lt 0) then return, 0
  if (colorsOut.ndim ne 2) then begin
    if (colorsOut.length ne 3) then return, 0
  endif
  
  if (colorsOut.ndim eq 1) then colorsOut = reform(colorsOut, 3, 1)
  if ((colorsOut.dim)[0] ne 3) then colorsOut = transpose(colorsOut)
  if ((colorsOut.dim)[0] ne 3) then return, 0
  
  colorsOut = byte(colorsOut)
  return, 1 
  
end

;----------------------------------------------------------------------------
;
function dialog_colorpickerDefaultColors
  compile_opt idl2, hidden

  return, [[255, 255, 255], $ ;Row 1
           [255, 204, 204], $
           [255, 237, 204], $
           [255, 255, 204], $
           [204, 255, 204], $
           [204, 204, 255], $
           [226, 213, 245], $
           [235, 223, 235], $
           [218, 218, 218], $ ;Row 2
           [255, 160, 160], $
           [255, 221, 160], $
           [255, 255, 160], $
           [160, 255, 160], $
           [160, 160, 255], $
           [201, 178, 236], $
           [219, 195, 219], $
           [182, 182, 182], $ ;Row 3
           [255, 116, 116], $
           [255, 206, 116], $
           [255, 255, 116], $
           [116, 255, 116], $
           [116, 116, 255], $
           [177, 142, 228], $
           [203, 168, 203], $ 
           [145, 145, 145], $ ;Row 4
           [255, 72, 72], $
           [255, 190, 72], $
           [255, 255, 72], $
           [72, 255, 72], $
           [72, 72, 255], $
           [152, 107, 220], $
           [187, 140, 187], $ 
           [109, 109, 109], $ ;Row 5
           [255, 29, 29], $
           [255, 175, 29], $
           [255, 255, 29], $
           [29, 255, 29], $
           [29, 29, 255], $
           [128, 72, 211], $
           [170, 113, 170], $ 
           [72, 72, 72], $ ;Row 6
           [240, 0, 0], $
           [240, 155, 0], $
           [240, 240, 0], $
           [0, 240, 0], $
           [0, 0, 240], $
           [105, 45, 194], $
           [150, 89, 150], $
           [36, 36, 36], $ ;Row 7
           [196, 0, 0], $
           [196, 127, 0], $
           [196, 196, 0], $
           [0, 196, 0], $
           [0, 0, 196], $
           [86, 37, 159], $
           [123, 73, 123], $
           [0, 0, 0], $ ;Row 8
           [153, 0, 0], $
           [153, 99, 0], $
           [153, 153, 0], $
           [0, 153, 0], $
           [0, 0, 153], $
           [67, 29, 123], $
           [95, 57, 95]]
           
end

;----------------------------------------------------------------------------
; dialog_colorpickerCleanup
;
pro dialog_colorpickerCleanup, wTop
  compile_opt idl2, hidden

  Widget_Control, wTop, GET_UVALUE=pState
  if (Ptr_Valid(pState)) then begin
    if (Widget_Info((*pState).wCaller, /VALID_ID)) then begin
      Widget_Control, (*pState).wCaller, SET_BUTTON=0
    endif
    if (~(*pState).block) then begin
      PTR_FREE, [(*pState).result, (*pState).customColors, (*pState).udata]
      PTR_FREE, pState
    endif
  endif  
end

;----------------------------------------------------------------------------
; dialog_colorpickerDestroy
;
pro dialog_colorpickerDestroy, sEvent
  compile_opt idl2, hidden

  Widget_Control, sEvent.top, /DESTROY
end

;----------------------------------------------------------------------------
; dialog_colorpickerSwitchToCustom
;
pro dialog_colorpickerSwitchToCustom, sEvent
  compile_opt idl2, hidden

  Widget_Control, sEvent.top, GET_UVALUE=pState
  Widget_Control, (*pState).wTab, SET_TAB_CURRENT=1
end

;----------------------------------------------------------------------------
; dialog_colorpickerUpdateLuminosityPick
;
pro dialog_colorpickerUpdateLuminosityPick, wTop
  compile_opt idl2, hidden

  Widget_Control, wTop, GET_UVALUE=pState
  Widget_Control, (*pState).wL, GET_VALUE=luminosity
  Widget_Control, (*pState).wPickRamp, GET_VALUE=wID
  geom = Widget_Info((*pState).wPickRamp, /GEOMETRY)
  x = Fix(Fix(luminosity)/255.0*geom.xSize)
  img = BytArr(geom.xSize, geom.ySize, 3)
  img[*,*,0] = (*pState).face3D[0]
  img[*,*,1] = (*pState).face3D[1]
  img[*,*,2] = (*pState).face3D[2]
  sz = Size((*pState).bitmapPicker, /DIMENSIONS)
  minX = x-sz[0]/2
  if (minX lt 0) then begin
    minX = 0
    minPickX = sz[0]/2-x
  endif else begin
    minPickX = 0
  endelse
  maxX = x+sz[0]/2+1 < geom.xSize-1
  diff = maxX-minX
  offset = ((*pState).face3D[0] ge 128) ? 0 : 255 
  img[minX:maxX,*,0] = (*pState).bitmapPicker[minPickX:minPickX+diff,*,0] + offset
  img[minX:maxX,*,1] = (*pState).bitmapPicker[minPickX:minPickX+diff,*,1] + offset
  img[minX:maxX,*,2] = (*pState).bitmapPicker[minPickX:minPickX+diff,*,2] + offset
  win = !d.window
  Wset, wID
  Device, GET_DECOMPOSED=dec
  Device, DECOMPOSED=0
  Tv, img, TRUE=3, /ORDER
  Device, DECOMPOSED=dec
  wSet, win
end

;----------------------------------------------------------------------------
; dialog_colorpickerUpdateLuminosityRamp
;
pro dialog_colorpickerUpdateLuminosityRamp, wTop
  compile_opt idl2, hidden

  Widget_Control, wTop, GET_UVALUE=pState
  Widget_Control, (*pState).wH, GET_VALUE=h
  Widget_Control, (*pState).wS, GET_VALUE=s
  
  geom = Widget_Info((*pState).wLuminosity, /GEOMETRY)
  hRamp = FltArr(geom.xSize)+Fix(h[0])/255.0*360
  sRamp = FltArr(geom.xSize)+Fix(s[0])/255.0
  lRamp = FindGen(geom.xSize)/(geom.xSize-1)
  hValues = hRamp # (FltArr(geom.ySize)+1.0)
  sValues = sRamp # (FltArr(geom.ySize)+1.0)
  lValues = lRamp # (FltArr(geom.ySize)+1.0)
  Color_Convert, [[[hValues]], [[lValues]], [[sValues]]], rgbImage, $
                 /HLS_RGB, INTERLEAVE=2    
  Widget_Control, (*pState).wLuminosity, GET_VALUE=wID
  win = !d.window
  Device, GET_DECOMPOSED=dec
  Wset, wID
  Device, DECOMPOSED=0
  Tv, rgbImage, TRUE=3
  Device, DECOMPOSED=dec
  Wset, win  
  dialog_colorpickerUpdateLuminosityPick, wTop
end

;----------------------------------------------------------------------------
; dialog_colorpickerUpdateColorSpacePick
;
pro dialog_colorpickerUpdateColorSpacePick, wTop
  compile_opt idl2, hidden

  Widget_Control, wTop, GET_UVALUE=pState
  Widget_Control, (*pState).wColorSpace, GET_VALUE=wID
  Widget_Control, (*pState).wS, GET_VALUE=s
  Widget_Control, (*pState).wH, GET_VALUE=h
  img = (*pState).colorSpaceBitmap
  sz = Size(img, /DIMENSIONS)
  x = Fix(h)/255.0*sz[0]
  y = Fix(s)/255.0*sz[1]  
  ; Crosshair shape
  Widget_Control, (*pState).wL, GET_VALUE=luminosity
  pickColor = (Fix(luminosity) ge 100) ? 0b : 255b  
  for i=2, 10 do begin
    img[x-i > 0, y, *] = pickColor
    img[x, y-i > 0, *] = pickColor
    img[x+i+1 < sz[0]-1, y, *] = pickColor
    img[x, y+i+1 < sz[1]-1, *] = pickColor
  endfor
  win = !d.window
  Wset, wID
  Device, GET_DECOMPOSED=dec
  Device, DECOMPOSED=0
  Tv, img, TRUE=3
  Device, DECOMPOSED=dec
  wSet, win
end

;----------------------------------------------------------------------------
; dialog_colorpickerUpdateColorSpace
;
pro dialog_colorpickerUpdateColorSpace, wTop
  compile_opt idl2, hidden

  Widget_Control, wTop, GET_UVALUE=pState
  Widget_Control, (*pState).wL, GET_VALUE=l
  geom = Widget_Info((*pState).wColorSpace, /GEOMETRY)
  
  ; Create custom color map of hues and saturations.
  hRamp = FindGen(geom.xSize)/Float(geom.xSize-1)*360.0
  sRamp = FindGen(geom.ySize)/Float(geom.ySize-1)
  lRamp = FltArr(geom.xSize)+Fix(l[0])/255.0
  hValues = hRamp # (FltArr(geom.ySize)+1.0)
  sValues = sRamp ## (FltArr(geom.xSize)+1.0)
  lValues = lRamp # (FltArr(geom.ySize)+1.0)
  Color_Convert, [[[hValues]], [[lValues]], [[sValues]]], colorSpaceBitmap, $
                 /HLS_RGB, INTERLEAVE=2
  (*pState).colorSpaceBitmap = colorSpaceBitmap
  ; We don't need to draw the image here, updating the pick will do it. 
  dialog_colorpickerUpdateColorSpacePick, wTop  
end

;----------------------------------------------------------------------------
; dialog_colorpickerTipText
;
function dialog_colorpickerTipText, r, g, b
  compile_opt idl2, hidden
  
  return, ( !version.os_family eq 'Windows' ) ? $
    r.toString() + ',' + g.toString() + ',' + b.toString() : $
    !null
end

;----------------------------------------------------------------------------
; dialog_colorpickerUpdateSelected
;
pro dialog_colorpickerUpdateSelected, wTop
  compile_opt idl2, hidden

  Widget_Control, wTop, GET_UVALUE=pState
  Widget_Control, (*pState).wSelected, GET_VALUE=wID
  Widget_Control, (*pState).wR, GET_VALUE=r
  Widget_Control, (*pState).wG, GET_VALUE=g
  Widget_Control, (*pState).wB, GET_VALUE=b
  win = !d.window
  Device, GET_DECOMPOSED=dec
  geom = Widget_Info((*pState).wSelected, /GEOMETRY)
  img = BytArr(geom.xSize, geom.ySize, 3)
  img[*,*,0] = Fix(r)
  img[*,*,1] = Fix(g)
  img[*,*,2] = Fix(b)
  Wset, wID
  Device, DECOMPOSED=0
  Tv, img, TRUE=3
  Device, DECOMPOSED=dec
  Wset, win
  Widget_Control, (*pState).wSelected, TOOLTIP=dialog_colorpickerTipText( r, g, b )
end

;----------------------------------------------------------------------------
; dialog_colorpickerUpdateRGB
;
pro dialog_colorpickerUpdateRGB, wTop
  compile_opt idl2, hidden

  Widget_Control, wTop, GET_UVALUE=pState
  Widget_Control, (*pState).wH, GET_VALUE=h
  Widget_Control, (*pState).wS, GET_VALUE=s
  Widget_Control, (*pState).wL, GET_VALUE=l
  
  Color_Convert, h/255.0*360, l/255.0, s/255.0, r, g, b, /HLS_RGB
  
  Widget_Control, (*pState).wR, SET_VALUE=StrTrim(Fix(r), 2)
  Widget_Control, (*pState).wG, SET_VALUE=StrTrim(Fix(g), 2)
  Widget_Control, (*pState).wB, SET_VALUE=StrTrim(Fix(b), 2)
  
  dialog_colorpickerUpdateColorSpace, wtop
  dialog_colorpickerUpdateSelected, wTop
  Widget_Control, (*pState).wOK, SET_UVALUE=[Fix(r), Fix(g), Fix(b)]
end

;----------------------------------------------------------------------------
; dialog_colorpickerUpdateHLS
;
pro dialog_colorpickerUpdateHLS, wTop
  compile_opt idl2, hidden

  Widget_Control, wTop, GET_UVALUE=pState
  Widget_Control, (*pState).wR, GET_VALUE=r
  Widget_Control, (*pState).wG, GET_VALUE=g
  Widget_Control, (*pState).wB, GET_VALUE=b

  Color_Convert, Fix(r), Fix(g), Fix(b), h, l, s, /RGB_HLS

  Widget_Control, (*pState).wH, SET_VALUE=StrTrim(Fix(h/360.0*255), 2)
  Widget_Control, (*pState).wS, SET_VALUE=StrTrim(Fix(s*255), 2)
  Widget_Control, (*pState).wL, GET_VALUE=oldL
  Widget_Control, (*pState).wL, SET_VALUE=StrTrim(Fix(l*255), 2)
  dialog_colorpickerUpdateColorSpace, wtop
  dialog_colorpickerUpdateSelected, wTop
  Widget_Control, (*pState).wOK, SET_UVALUE=[Fix(r), Fix(g), Fix(b)]
end

;----------------------------------------------------------------------------
; dialog_colorpickerUpDownEvent
;
pro dialog_colorpickerUpDownEvent, sEvent
  compile_opt idl2, hidden

  Widget_Control, sEvent.top, GET_UVALUE=pState
  uname = StrTok(Widget_Info(sEvent.id, /UNAME), '-', /EXTRACT)
  color = uname[0]
  if (color eq 'current') then begin
    ; This was invoked via accelerator, figure out currently focused variable.
    color = (*pState).currentVariable
    if (color eq '') then begin
      return
    endif
  endif
  direction = uname[1]
  wText = Widget_Info(sEvent.top, FIND_BY_UNAME=color)
  if (~Widget_Info(wText, /VALID_ID)) then begin
    return
  endif
  Widget_Control, wText, GET_VALUE=value
  if (direction eq 'up') then begin
    value = Fix(value)+1
  endif else begin
    value = Fix(value)-1
  endelse
  value = 0 > value < 255
  ; Back to widget element
  Widget_Control, wText, SET_VALUE=StrTrim(value, 2)
  ; Update color
  uname = Widget_Info(wText, /UNAME)
  if ((uname eq 'r') || (uname eq 'g') || (uname eq 'b')) then begin
    dialog_colorpickerUpdateHLS, sEvent.top
  endif else begin
    dialog_colorpickerUpdateRGB, sEvent.top
  endelse
  dialog_colorpickerUpdateLuminosityRamp, sEvent.top
end

;----------------------------------------------------------------------------
; dialog_colorpickerTextEvent
;
pro dialog_colorpickerTextEvent, sEvent
  compile_opt idl2, hidden

  Widget_Control, sEvent.top, GET_UVALUE=pState
  eventName = Tag_Names(sEvent, /STRUCTURE_NAME)
  if (eventName eq 'WIDGET_KBRD_FOCUS') then begin
    if (sEvent.enter) then begin
      Widget_Control, sEvent.id, SET_TEXT_SELECT=[0,3]
      (*pState).currentVariable = Widget_Info(sEvent.id, /UNAME)
    endif else begin
      Widget_Control, sEvent.id, SET_TEXT_SELECT=[0,0]
      (*pState).currentVariable = ''
    endelse 
    return
  endif  
  if ((eventName eq 'WIDGET_TEXT_CH') || $
      (eventName eq 'WIDGET_TEXT_DEL')) then begin
    Widget_Control, sEvent.id, GET_VALUE=str
    selectLocation = Widget_Info(sEvent.id, /TEXT_SELECT)
    ; make it numeric
    bStr = Byte(str)
    numericStr = List()
    for i=0, N_Elements(bStr)-1 do begin
      if ((bStr[i] ge 48b) && (bStr[i] le 57b)) then begin
        numericStr.Add, String(bStr[i]) 
      endif else begin
        selectLocation = 0
      endelse
    endfor
    if (numericStr.Count() eq 0) then begin
      numericStr.Add, '0'
    endif
    numericVal = Fix(StrJoin(numericStr.ToArray()))
    ; Apply thresholds 
    numericVal = 0 > numericVal < 255
    ; Back to widget element
    Widget_Control, sEvent.id, SET_VALUE=StrTrim(numericVal, 2)
    Widget_Control, sEvent.id, SET_TEXT_SELECT=[selectLocation[0],0]
    Obj_Destroy, numericStr
    ; Update color
    uname = Widget_Info(sEvent.id, /UNAME)
    if ((uname eq 'r') || (uname eq 'g') || (uname eq 'b')) then begin
      dialog_colorpickerUpdateHLS, sEvent.top
    endif else begin
      dialog_colorpickerUpdateRGB, sEvent.top
    endelse
  endif
  dialog_colorpickerUpdateLuminosityRamp, sEvent.top
end

;----------------------------------------------------------------------------
; dialog_colorpickerLuminosityEvent
;
pro dialog_colorpickerLuminosityEvent, sEvent
  compile_opt idl2, hidden

  Widget_Control, sEvent.top, GET_UVALUE=pState
  eventName = Tag_Names(sEvent, /STRUCTURE_NAME)
  if (eventName eq 'WIDGET_TRACKING') then begin
    Widget_Control, (*pState).wPickRamp, GET_VALUE=wID
    win = !d.window
    Wset, wID
    if (sEvent.enter) then begin
      Device, CURSOR_STANDARD=(!version.os_family ne 'Windows' ? 22 : 32512)
    endif else begin
      Device, /CURSOR_CROSSHAIR
    endelse
    Wset, win
    return
  endif
  
  if (sEvent.press eq 1) then begin
    (*pState).mouseDownLuminosity = 1
  endif
  if (sEvent.release eq 1) then begin
    (*pState).mouseDownLuminosity = 0
  endif
  if ((*pState).mouseDownLuminosity) then begin
    geom = Widget_Info((*pState).wPickRamp, /GEOMETRY)
    if ((sEvent.x lt 0) || (sEvent.x gt geom.xSize-1)) then begin
      return
    endif
    newL = StrTrim(Fix(sEvent.x/Float(geom.xSize-1)*255), 2)
    Widget_Control, (*pState).wL, GET_VALUE=oldL
    Widget_Control, (*pState).wL, SET_VALUE=newL
    if (oldL ne newL) then begin
      dialog_colorpickerUpdateColorSpace, sEvent.top
    endif
    dialog_colorpickerUpdateLuminosityPick, sEvent.top
    dialog_colorpickerUpdateRGB, sEvent.top
  endif 
end

;----------------------------------------------------------------------------
; dialog_colorpickerColorSpaceEvent
;
pro dialog_colorpickerColorSpaceEvent, sEvent
  compile_opt idl2, hidden

  Widget_Control, sEvent.top, GET_UVALUE=pState
      
  eventName = Tag_Names(sEvent, /STRUCTURE_NAME)
  if (eventName eq 'WIDGET_TRACKING') then begin
    win = !d.window
    Widget_Control, (*pState).wColorSpace, GET_VALUE=wID
    Wset, wID
    if (sEvent.enter) then begin
      Device, CURSOR_STANDARD=(!version.os_family ne 'Windows' ? 22 : 32512)
    endif else begin
      Device, /CURSOR_CROSSHAIR
    endelse
    Wset, win
    return
  endif  
  
  if (sEvent.press eq 1) then begin
    (*pState).mouseDownColor = 1
  endif
  if (sEvent.release eq 1) then begin
    (*pState).mouseDownColor = 0
  endif 
  if ((*pState).mouseDownColor) then begin
    geom = Widget_Info((*pState).wColorSpace, /GEOMETRY)
    x = 0 > sEvent.x < (geom.xSize-1)
    y = 0 > sEvent.y < (geom.ySize-1)
    Widget_Control, (*pState).wH, $
                    SET_VALUE=StrTrim(Fix(x/Float(geom.xSize-1)*255), 2)
    Widget_Control, (*pState).wS, $
                    SET_VALUE=StrTrim(Fix(y/Float(geom.ySize-1)*255), 2)
    dialog_colorpickerUpdateRGB, sEvent.top
    dialog_colorpickerUpdateColorSpacePick, sEvent.top
    dialog_colorpickerUpdateLuminosityRamp, sEvent.top
    if (sEvent.clicks eq 2) then begin
      sEvent = {WIDGET_BUTTON, $
                id: (*pState).wOK, $
                top: sEvent.top, $
                handler: (*pState).wOK, $
                select: 1}
      dialog_colorpickerEvent, sEvent 
    endif
  endif
end

;----------------------------------------------------------------------------
; dialog_colorpickerEvent
;
pro dialog_colorpickerEvent, sEvent
  compile_opt idl2, hidden
  
  case (Tag_Names(sEvent, /STRUCTURE_NAME)) of  
    'WIDGET_KBRD_FOCUS': begin
      if (sEvent.enter eq 0) then begin
        Widget_Control, sEvent.top, /DESTROY
      endif      
    end
    'WIDGET_BUTTON': begin
      Widget_Control, sEvent.top, GET_UVALUE=pState
      wTop = sEvent.top
      wCaller = (*pState).wCaller
      wTlb = (*pState).wTlb
      Widget_Control, sEvent.id, GET_UVALUE=color
      customColors = *(*pState).customColors
      if (Widget_Info(sEvent.id, /UNAME) eq 'Custom_OK') then begin
        ; It was a custom color, add to preferences
        customColors = [[customColors], [color]]
        ; Let's have a max number of stored colors so the list doesn't grow forever.
        maxColors = 20
        if (N_ELEMENTS(customColors) / 3 gt maxColors) then begin
          customColors = customColors[*,0:maxColors-1]
        endif
      endif
      routine = (*pState).callback
      customIsList = (*pState).customIsList
      userdata = ISA(*(*pState).udata) ? *(*pState).udata : !NULL 
      Widget_Control, wTop, /DESTROY
      if (customIsList) then begin
        customColorsOut = list()
        nCCs = N_ELEMENTS(customColors) / 3
        for i=0,nCCs-1 do begin
          customColorsOut.add, customColors[*,i]
        endfor
      endif else begin
        customColorsOut = customColors
      endelse
      if (N_ELEMENTS(customColorsOut) eq 0) then begin
        customColorsOut = list()
      endif
      sEvent = {ID: wCaller, $
        TOP: wTlb, $
        HANDLER: wCaller, $
        VALUE: color, $
        CUSTOMCOLORS: customColorsOut}
      if (routine ne '') then begin
        ; Call callback with user data if supplied
        if ( ISA(userdata) && $
          ((routine_info(routine, /PARAMETERS)).num_args gt 1) ) then begin
          Call_Procedure, routine, sEvent, userdata
        endif else begin
          Call_Procedure, routine, sEvent
        endelse
      endif else begin
        ; if no call back, fill in result so data can be returned
        *(*pState).result = sEvent
      endelse
    end
    else: ;ignore
  end
end

;----------------------------------------------------------------------------
; dialog_colorpicker
;
function dialog_colorpicker, $
  WCALLER=wCallerIn, $
  DIALOG_PARENT=dialogParentIn, $
  CALLBACK=callbackIn, $
  NCOLS=nColsIn, $
  NROWS=nRowsIn, $
  INITIAL_COLOR=initialColor, $
  COLORS=colorsIn, $
  MODAL=modalIn, $
  BOX_SIZE=boxSizeIn, $
  CUSTOM_COLORS=customColorsIn, $
  CUSTOM_OUT=customOut, $
  CUSTOM_TITLE=customTitleIn, $
  PREFERRED_COLORS=prefColorsIn, $
  PREFERRED_TITLE=prefTitleIn, $
  TITLE=titleIn, $
  USER_DATA=udata, $
  TAB_ON_OPEN=tabOnOpenIn, $
  STRINGS=strHash, $
  FRAMELESS=frameless, $
  XOFFSET=xoffsetIn, $
  YOFFSET=yoffsetIn
compile_opt idl2, hidden
on_error, 2

  if !magic.embed eq 1 then message,NAME='IDL_M_WIDGET_NODISPLAY'
  
  isWindows = (!version.os_family eq 'Windows')

  modal = 0b
  if (ISA(wCallerIn)) then begin
    if ( (wCallerIn.length eq 1) && ISA(wCallerIn, /NUMBER) ) then begin
      wCaller = wCallerIn
      ; Find top level base
      wTop = wCaller
      while (WIDGET_INFO(WIDGET_INFO(wTop, /PARENT), /VALID_ID)) do begin
        wTop = WIDGET_INFO(wTop, /PARENT)
      endwhile
      modal = WIDGET_INFO(wTop, /MODAL) ? 1 : KEYWORD_SET(modalIn)
    endif else begin
      message, 'Invalid value for wCaller'
    endelse
  endif else begin
    wCaller = 0
  endelse
  
  hasInitialColor = 0b
  if (N_Elements(initialColor) ne 0) then begin
    if ((N_Elements(initialColor) ne 3) || $
        (Where(initialColor gt 255, /null) ne !null) || $
        (Where(initialColor lt 0, /null) ne !null)) then begin
      message, 'Invalid value for INITIAL_COLOR'
    endif else begin
      hasInitialColor = 1b
    endelse
  endif
   
  if (ISA(dialogParentIn)) then begin
    if ( (dialogParentIn.LENGTH eq 1) && ISA(dialogParentIn, /NUMBER) ) then begin
      dialogParent = dialogParentIn
    endif else begin
      message, 'Invalid value for DIALOG_PARENT'
    endelse
  endif else begin
    dialogParent = 0
  endelse

  if (ISA(titleIn)) then begin
    if ( (titleIn.LENGTH eq 1) && (ISA(titleIn, /STRING)) ) then begin
      title = titleIn
    endif else begin
      message, 'Invalid value for TITLE'
    endelse
  endif else begin
    title = 'Select Color'
  endelse

  if (ISA(callbackIn)) then begin
    catch, err
    if (err ne 0) then begin
      catch, /cancel
      message, 'Callback routine does not exist'
    endif
    if ( (callbackIn.length eq 1) && (ISA(callbackIn, /STRING)) ) then begin
      !NULL = routine_info(callbackIn, /PARAMETERS)
      callback = callbackIn
      block = 0b
    endif else begin
      message, 'Invalid value for CALLBACK'
    endelse
  endif else begin
    callback = ''
    block = 1b
  endelse
  catch, /cancel
  
  if (ISA(boxSizeIn)) then begin
    if ( (boxSizeIn.LENGTH eq 1) && ISA(boxSizeIn, /NUMBER) && $
        (boxSizeIn gt 5) ) then begin
      boxSize = boxSizeIn
    endif else begin
      message, 'Invalid value for boxSize'
    endelse
  endif else begin
    boxSize = (isWindows) ? 19 : 24
  endelse
  
  if (ISA(colorsIn)) then begin
    if (dialog_colorpickerValidateColors(colorsIn, colorsOut)) then begin
      colors = colorsOut
    endif else begin
      message, 'Invalid COLORS'
    endelse
  endif else begin
    colors = dialog_colorpickerDefaultColors()
  endelse
  nColors = (colors.dim)[1]
  
  customIsList = 0b
  if (ISA(customColorsIn)) then begin
    customIsList = ISA(customColorsIn, 'list')
    if (dialog_colorpickerValidateColors(customColorsIn, customColorsOut)) then begin
      customColors = customColorsOut
    endif else begin
      message, 'Invalid CUSTOM_COLORS'
    endelse
  endif else begin
    customColors = []
  endelse

  if (ISA(customTitleIn)) then begin
    if ( (customTitleIn.LENGTH eq 1) && (ISA(customTitleIn, /STRING)) ) then begin
      customTitle = customTitleIn
    endif else begin
      message, 'Invalid value for CUSTOM_TITLE'
    endelse
  endif else begin
    customTitle = 'Custom Colors'
  endelse

  if (ISA(prefTitleIn)) then begin
    if ( (prefTitleIn.LENGTH eq 1) && (ISA(prefTitleIn, /STRING)) ) then begin
      prefTitle = prefTitleIn
    endif else begin
      message, 'Invalid value for PREFERRED_TITLE'
    endelse
  endif else begin
    prefTitle = 'Preferred Colors'
  endelse

  if (ISA(prefColorsIn)) then begin
    if (dialog_colorpickerValidateColors(prefColorsIn, prefColorsOut)) then begin
      prefColors = prefColorsOut
    endif else begin
      message, 'Invalid PREFERRED_COLORS'
    endelse
  endif else begin
    prefColors = []
  endelse

  if (ISA(tabOnOpenIn)) then begin
    if ( (tabOnOpenIn.LENGTH eq 1) && ISA(tabOnOpenIn, /NUMBER) && $
      (tabOnOpenIn ge 0) ) then begin
      tabOnOpen = tabOnOpenIn
    endif else begin
      message, 'Invalid value for TAB_ON_OPEN'
    endelse
  endif else begin
    tabOnOpen = 0
  endelse
  
  hasStrings = ( ISA(strHash, 'hash') && (strHash.length ge 10) )

  if (ISA(nColsIn)) then begin
    if ( (nColsIn.LENGTH eq 1) && ISA(nColsIn, /NUMBER) && $
      (nColsIn gt 0) ) then begin
      nCols = nColsIn
    endif else begin
      message, 'Invalid value for nCols'
    endelse
  endif else begin
    nCols = 8
  endelse

  if (ISA(nRowsIn)) then begin
    if ( (nRowsIn.LENGTH eq 1) && ISA(nRowsIn, /NUMBER) && $
      (nRowsIn gt 0) ) then begin
      nRows = nRowsIn
    endif else begin
      message, 'Invalid value for nRows'
    endelse
  endif else begin
    nRows = ceil(nColors/float(nCols))
  endelse

  xSizeColorMap = nCols*boxSize
  ySizeColorMap = Fix(xSizeColorMap*0.8)
  
  ; Compute location for the widget  
  offset = [0,0]
  offsetAdjust = (isWindows) ? 6 : 0
  accumulatedOffset = 2
  if (ISA(xoffsetIn) && ISA(yoffsetIn)) then begin
    xoffset = xoffsetIn
    yoffset = yoffsetIn
    geom = {WIDGET_GEOMETRY}
    wTlb = 0l
  endif else begin
    if (widget_info(wCaller, /VALID_ID)) then begin
      !null = dialog_colorpickerGetAccumulatedWidgetOffset(wCaller, OFFSET=offset, TLB=wTlb)
      geom = Widget_Info(wCaller, /GEOMETRY)
      xoffset = offset[0]-offsetAdjust
      yoffset = offset[1]+geom.SCR_YSIZE-offsetAdjust
    endif else begin
      geom = {WIDGET_GEOMETRY}
      wTlb = 0l
      xoffset = 0
      yoffset = 0
    endelse
  endelse
  
  showFrame = (KEYWORD_SET(frameless) || (wTLB ne 0)) ? 0 : block
  groupLeader = modal ? wCaller : dialogParent
  if (groupLeader gt 0) then begin
    leaderModal = WIDGET_INFO(groupLeader, /MODAL)
    if (leaderModal) then begin
      showFrame = 1
    endif
    ; In Unix, we need to ensure that this widget is a base.
    if (~isWindows && (WIDGET_INFO(groupLeader, /TYPE) ne 0)) then begin
      groupLeader = WIDGET_INFO(groupLeader, /PARENT)
    endif
  endif

  if (KEYWORD_SET(modal)) then showFrame = 1b
  ; NOTE: Unix shows a bar on this base... must have a title. 
  wBase = Widget_Base(XPAD=0, YPAD=0, SPACE=0, $
                      /KBRD_FOCUS_EVENTS, $
                      TLB_FRAME_ATTR=showFrame ? 1 : 31, $
                      TITLE=title, $
                      UNAME='tlb', $
                      XOFFSET=modal ? !NULL : xoffset, $
                      YOFFSET=modal ? !NULL : yoffset, $
                      MODAL=modal, $
                      GROUP_LEADER=groupLeader, $
                      KILL_NOTIFY='dialog_colorpickerCleanup')
  wTab = Widget_Tab(wBase, XOFFSET=0)
  key = 'Standard'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key 
  wBaseStandard = Widget_Base(wTab, TITLE=str)                                                    

  img = BytArr(boxSize, boxSize, 4)
  img[*,*,3] = 255b

  if (N_ELEMENTS(prefColors) ne 0) then begin
    wLabel = Widget_Label(wBaseStandard, $
                          VALUE=prefTitle, $
                          XOFFSET=2, $
                          YOFFSET=accumulatedOffset)
    accumulatedOffset += (Widget_Info(wLabel, /GEOMETRY)).scr_ysize + 1
    
    nPref = nCols < (prefColors.dim)[1]
    for j=0,nPref-1 do begin
      img[*,*,0] = prefColors[0,j]
      img[*,*,1] = prefColors[1,j]
      img[*,*,2] = prefColors[2,j]
      if (hasInitialColor && Array_Equal(initialColor, prefColors[*,j])) then begin
        dialog_colorpickerHighlightBitmap, img
      endif
      !null = Widget_Button(wBaseStandard, VALUE=img, $
                            UVALUE=Reform(prefColors[*,j]), $
                            XOFFSET=(boxSize+1)*j, $
                            YOFFSET=accumulatedOffset, $
                            SCR_XSIZE=boxSize, $
                            SCR_YSIZE=boxSize, $
                            UNAME='color', $
                            TOOLTIP=dialog_colorpickerTipText( $
                              prefColors[0,j], prefColors[1,j], prefColors[2,j]), $
                            /FLAT)
    endfor
    accumulatedOffset += boxSize+1
  endif

  key = 'System Colors'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  wLabel = Widget_Label(wBaseStandard, $
                        VALUE=str, $
                        XOFFSET=2, $
                        YOFFSET=accumulatedOffset) 
  accumulatedOffset += (Widget_Info(wLabel, /GEOMETRY)).scr_ysize

  ; System Colors
  n = 0
  for i=0,nRows-1 do begin
    yOffset = (boxSize+1)*i + accumulatedOffset
    for j=0,nCols-1 do begin
      xOffset = (boxSize+1)*j
      img[*,*,0] = colors[0,n]
      img[*,*,1] = colors[1,n]
      img[*,*,2] = colors[2,n]
      if (hasInitialColor && Array_Equal(initialColor, colors[*,n])) then begin
        dialog_colorpickerHighlightBitmap, img
      endif
      !null = Widget_Button(wBaseStandard, VALUE=img, $
                                   UVALUE=Reform(colors[*,n]), $
                                   XOFFSET=xOffset, $
                                   YOFFSET=yOffset, $
                                   SCR_XSIZE=boxSize, $
                                   SCR_YSIZE=boxSize, $
                                   UNAME='color', $
                                   TOOLTIP=dialog_colorpickerTipText( $
                                     colors[0,n], colors[1,n], colors[2,n]), $
                                   /FLAT)
      n++
      if (n ge nColors) then break
    endfor
    if (n ge nColors) then break
  endfor

  accumulatedOffset = yOffset + (boxSize-1) ;- ((isWindows) ? 1 : 7)
  wLabel = Widget_Label(wBaseStandard, $
                        VALUE=CustomTitle, $
                        XOFFSET=2, $
                        YOFFSET=accumulatedOffset)
  accumulatedOffset += (Widget_Info(wLabel, /GEOMETRY)).scr_ysize

  ; Custom colors - Part I, defined colors
  if (N_ELEMENTS(customColors) ne 0) then begin
    nCustom = customColors.NDIM eq 2 ? (customColors.dim)[1] : 1
  endif else begin
    nCustom = 0
  endelse
  for j=0, (nCustom < nCols)-1 do begin
    ; Display in reverser order so the last one is always on the left.
    customColor = customColors[*,nCustom-1-j]
    img[*,*,0] = customColor[0]
    img[*,*,1] = customColor[1]
    img[*,*,2] = customColor[2]
    if (hasInitialColor && Array_Equal(initialColor, customColor)) then begin
      dialog_colorpickerHighlightBitmap, img
    endif
    !null = Widget_Button(wBaseStandard, VALUE=img, $
                          UVALUE=Reform(customColor), $
                          XOFFSET=(boxSize+1)*j, $
                          YOFFSET=accumulatedOffset, $
                          SCR_XSIZE=boxSize, $
                          SCR_YSIZE=boxSize, $
                          UNAME='color', $
                          TOOLTIP=dialog_colorpickerTipText( $
                            customColor[0], customColor[1], customColor[2]), $
                          /FLAT)
  endfor

  ; Custom colors - Part II, empty slots
  img[*,*,0] = 255B
  img[*,*,1] = 255B
  img[*,*,2] = 255B
  key = 'Define Custom Color...'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  tooltip = ( !version.os_family eq 'Windows' ? str : !null )
  for j= nCustom<nCols, nCols-1 do begin
    !null = Widget_Button(wBaseStandard, $
                          VALUE=img, $
                          XOFFSET=(boxSize+1)*j, $
                          YOFFSET=accumulatedOffset, $
                          SCR_XSIZE=boxSize, $
                          SCR_YSIZE=boxSize, $
                          UNAME='select_custom', $
                          TOOLTIP=toolTip, $
                          EVENT_PRO='dialog_colorpickerSwitchToCustom', $
                          /FLAT)
  endfor

  ; Custom selection
  key = 'Custom'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  wBaseCustom = Widget_Base(wTab, $
                            /COLUMN, $
                            TITLE=str)  
  sz = Size(colorSpaceBitmap, /DIMENSIONS)
  wBaseCustomDraw = Widget_Base(wBaseCustom, $
                                /ROW, $
                                XPAD=0, YPAD=0, $
                                /ALIGN_CENTER)
  wColorSpace = Widget_Draw(wBaseCustomDraw, RETAIN=2, $
                                             XSIZE=xSizeColorMap, $
                                             YSIZE=ySizeColorMap, $
                                             EVENT_PRO='dialog_colorpickerColorSpaceEvent', $
                                             /TRACKING_EVENTS, $
                                             /BUTTON_EVENTS, $
                                             /MOTION_EVENTS)
  wBaseCustomDraw = Widget_Base(wBaseCustom, $
                                /ROW, $
                                XPAD=0, YPAD=0, $
                                /ALIGN_CENTER)
  wPickRamp = Widget_Draw(wBaseCustomDraw, RETAIN=2, $
                                           XSIZE=xSizeColorMap, $
                                           YSIZE=6, $
                                           /TRACKING_EVENTS, $
                                           /BUTTON_EVENTS, $
                                           /MOTION_EVENTS, $
                                           EVENT_PRO='dialog_colorpickerLuminosityEvent')
  wBaseCustomDraw = Widget_Base(wBaseCustom, $
                                /ROW, $
                                XPAD=0, YPAD=0, $
                                /ALIGN_CENTER)
  wLuminosity = Widget_Draw(wBaseCustomDraw, RETAIN=2, $
                                             XSIZE=xSizeColorMap, $
                                             YSIZE=16, $
                                             /TRACKING_EVENTS, $
                                             /BUTTON_EVENTS, $
                                             /MOTION_EVENTS, $
                                             EVENT_PRO='dialog_colorpickerLuminosityEvent')
  
  strName = 'arrow_pick_up'
  fileName = Filepath(strName+'.png',ROOT_DIR=!DIR, SUBDIRECTORY=['resource','bitmaps'])
  bitmapUp = READ_PNG(fileName)
  bitmapUp = TRANSPOSE(bitmapUp, [1,2,0])
  strName = 'arrow_pick_down'
  fileName = Filepath(strName+'.png',ROOT_DIR=!DIR, SUBDIRECTORY=['resource','bitmaps'])
  bitmapDown = READ_PNG(fileName)
  bitmapDown = TRANSPOSE(bitmapDown, [1,2,0])

  spaceSize = 4
  font = 'Helvetica*16'
  
  pickBaseXSize = (isWindows) ? 7 : 11
  spacerYSize = (isWindows) ? 3 : 7
  wRGBBase = Widget_Base(wBaseCustom, /ROW, XPAD=0, SPACE=0, /ALIGN_CENTER)
  wLabelBase = Widget_Base(wRGBBase, /COL, XPAD=0, YPAD=0, SPACE=0)
  !null = Widget_Base(wLabelBase, YSIZE=spacerYSize) ; Spacer
  key = 'R:'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  !null = Widget_Label(wLabelBase, $
                       VALUE=str, $
                       FONT=font)
  wR = Widget_Text(wRGBBase, VALUE='', $
                             XSIZE=3, $
                             /EDITABLE, $
                             UNAME='r', $
                             TAB_MODE=1, $
                             /KBRD_FOCUS_EVENTS, $
                             /ALL_EVENTS, $
                             EVENT_PRO='dialog_colorpickerTextEvent', $
                             FONT=font)
  wPickBase = Widget_Base(wRGBBase, /COLUMN, XPAD=0, YPAD=0, SPACE=0, $
                                    SCR_XSIZE=pickBaseXSize)
  !null = Widget_Button(wPickBase, VALUE=bitmapUp, $
                                   UNAME='r-up', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Button(wPickBase, VALUE=bitmapDown, $
                                   UNAME='r-down', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Base(wRGBBase, XSIZE=spaceSize) ; Spacer
  wLabelBase = Widget_Base(wRGBBase, /COL, XPAD=0, YPAD=0, SPACE=0)
  !null = Widget_Base(wLabelBase, YSIZE=spacerYSize) ; Spacer
  key = 'G:'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  !null = Widget_Label(wLabelBase, $
                       VALUE=str, $
                       FONT=font)
  wG = Widget_Text(wRGBBase, VALUE='', $
                             XSIZE=3, $
                             /EDITABLE, $
                             UNAME='g', $
                             TAB_MODE=1, $
                             /KBRD_FOCUS_EVENTS, $
                             /ALL_EVENTS, $
                             EVENT_PRO='dialog_colorpickerTextEvent', $
                             FONT=font)
  wPickBase = Widget_Base(wRGBBase, /COLUMN, XPAD=0, YPAD=0, SPACE=0, $
                                    SCR_XSIZE=pickBaseXSize)
  !null = Widget_Button(wPickBase, VALUE=bitmapUp, $
                                   UNAME='g-up', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Button(wPickBase, VALUE=bitmapDown, $
                                   UNAME='g-down', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Base(wRGBBase, XSIZE=spaceSize) ; Spacer
  wLabelBase = Widget_Base(wRGBBase, /COL, XPAD=0, YPAD=0, SPACE=0)
  !null = Widget_Base(wLabelBase, YSIZE=spacerYSize) ; Spacer
  key = 'B:'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  !null = Widget_Label(wLabelBase, $
                       VALUE=str, $
                       FONT=font)
  wB = Widget_Text(wRGBBase, VALUE='', $
                             XSIZE=3, $
                             /EDITABLE, $
                             UNAME='b', $
                             TAB_MODE=1, $
                             /KBRD_FOCUS_EVENTS, $
                             /ALL_EVENTS, $
                             EVENT_PRO='dialog_colorpickerTextEvent', $
                             FONT=font)
  wPickBase = Widget_Base(wRGBBase, /COLUMN, XPAD=0, YPAD=0, SPACE=0, $
                                    SCR_XSIZE=pickBaseXSize)
  !null = Widget_Button(wPickBase, VALUE=bitmapUp, $
                                   UNAME='b-up', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Button(wPickBase, VALUE=bitmapDown, $
                                   UNAME='b-down', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  wHSLBase = Widget_Base(wBaseCustom, /ROW, XPAD=0, SPACE=0, /ALIGN_CENTER)
  wLabelBase = Widget_Base(wHSLBase, /COL, XPAD=0, YPAD=0, SPACE=0)
  !null = Widget_Base(wLabelBase, YSIZE=spacerYSize) ; Spacer
  key = 'H:'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  !null = Widget_Label(wLabelBase, $
                       VALUE=str, $
                       FONT=font)
  wH = Widget_Text(wHSLBase, VALUE='', $
                             XSIZE=3, $
                             /EDITABLE, $
                             UNAME='h', $
                             TAB_MODE=1, $
                             /KBRD_FOCUS_EVENTS, $
                             /ALL_EVENTS, $
                             EVENT_PRO='dialog_colorpickerTextEvent', $
                             FONT=font)
  wPickBase = Widget_Base(wHSLBase, /COLUMN, XPAD=0, YPAD=0, SPACE=0, $
                                    SCR_XSIZE=pickBaseXSize)
  !null = Widget_Button(wPickBase, VALUE=bitmapUp, $
                                   UNAME='h-up', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Button(wPickBase, VALUE=bitmapDown, $
                                   UNAME='h-down', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Base(wHSLBase, XSIZE=spaceSize) ; Spacer
  wLabelBase = Widget_Base(wHSLBase, /COL, XPAD=0, YPAD=0, SPACE=0)
  !null = Widget_Base(wLabelBase, YSIZE=spacerYSize) ; Spacer
  key = 'S:'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  !null = Widget_Label(wLabelBase, $
                       VALUE=str, $
                       FONT=font)
  wS = Widget_Text(wHSLBase, VALUE='', $
                             XSIZE=3, $
                             /EDITABLE, $
                             UNAME='s', $
                             TAB_MODE=1, $
                             /KBRD_FOCUS_EVENTS, $
                             /ALL_EVENTS, $
                             EVENT_PRO='dialog_colorpickerTextEvent', $
                             FONT=font)
  wPickBase = Widget_Base(wHSLBase, /COLUMN, XPAD=0, YPAD=0, SPACE=0, $
                                    SCR_XSIZE=pickBaseXSize)
  !null = Widget_Button(wPickBase, VALUE=bitmapUp, $
                                   UNAME='s-up', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Button(wPickBase, VALUE=bitmapDown, $
                                   UNAME='s-down', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Base(wHSLBase, XSIZE=spaceSize) ; Spacer
  wLabelBase = Widget_Base(wHSLBase, /COL, XPAD=0, YPAD=0, SPACE=0)
  !null = Widget_Base(wLabelBase, YSIZE=spacerYSize) ; Spacer
  key = 'L:'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  !null = Widget_Label(wLabelBase, $
                       VALUE=str, $
                       FONT=font)
  wL = Widget_Text(wHSLBase, VALUE='128', $
                             XSIZE=3, $
                             /EDITABLE, $
                             UNAME='l', $
                             TAB_MODE=1, $
                             /KBRD_FOCUS_EVENTS, $
                             /ALL_EVENTS, $
                             EVENT_PRO='dialog_colorpickerTextEvent', $
                             FONT=font)
  wPickBase = Widget_Base(wHSLBase, /COLUMN, XPAD=0, YPAD=0, SPACE=0, $
                                    SCR_XSIZE=pickBaseXSize)
  !null = Widget_Button(wPickBase, VALUE=bitmapUp, $
                                   UNAME='l-up', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')
  !null = Widget_Button(wPickBase, VALUE=bitmapDown, $
                                   UNAME='l-down', $
                                   /FLAT, $
                                   EVENT_PRO='dialog_colorpickerUpDownEvent')  
  
  if (~isWindows) then begin
    !null = Widget_Base(wBaseCustom, YSIZE=4) ; Separator
  endif
  wButtonBase = Widget_Base(wBaseCustom, /ROW, /ALIGN_RIGHT, YPAD=0)
  key = 'Selected:'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  !null = Widget_Label(wButtonBase, $
                       VALUE=str, $
                       FONT=font)
  wSelectedBase = Widget_Base(wButtonBase, /COL, /ALIGN_CENTER)
  wSelected = Widget_Draw(wSelectedBase, RETAIN=2, $
                                         XSIZE=22, $
                                         YSIZE=22, $
                                         /TRACKING_EVENTS, $
                                         EVENT_PRO='dialog_colorpickerLuminosityEvent')
  wSeparator = Widget_Base(wButtonBase)
  
  ; Hidden up/down buttons
  !null = Widget_Button(wBase, VALUE=' ', $
                               SCR_XSIZE=1, $
                               SCR_YSIZE=1, $
                               EVENT_PRO='dialog_colorpickerUpDownEvent', $
                               UNAME='current-up', $
                               ACCELERATOR='Up')
  !null = Widget_Button(wBase, VALUE=' ', $
                               SCR_XSIZE=1, $
                               SCR_YSIZE=1, $
                               EVENT_PRO='dialog_colorpickerUpDownEvent', $
                               UNAME='current-down', $
                               ACCELERATOR='Down')
  
  wOKBase = Widget_Base(wButtonBase, /COLUMN, /ALIGN_CENTER)
  key = 'OK'
  str = hasStrings && strHash.hasKey(key) ? strHash[key] : key
  wOK = Widget_Button(wOKBase, VALUE=str, $                               
                               ACCELERATOR='return', $
                               UNAME='Custom_OK', $
                               XSIZE=60)                             
  dialog_colorpickerSpreadWidgets, wSeparator                                   
  
  ; Hidden escape button
  !null = Widget_Button(wBase, VALUE=' ', $
                               XOFFSET=1, $
                               YOFFSET=1, $
                               SCR_XSIZE=1, $
                               SCR_YSIZE=1, $
                               EVENT_PRO='dialog_colorpickerDestroy', $
                               ACCELERATOR='Escape')
  
  systemColors = Widget_Info(wBase, /SYSTEM_COLORS)
  bitmap = [[0,0,0,0,0,0,0,0,0], $
            [1,0,0,0,0,0,0,0,1], $
            [1,1,0,0,0,0,0,1,1], $
            [1,1,1,0,0,0,1,1,1], $
            [1,1,1,1,0,1,1,1,1], $
            [1,1,1,1,1,1,1,1,1]]
  bitmapPicker = [[[bitmap*systemColors.face_3d[0]]], $
                  [[bitmap*systemColors.face_3d[1]]], $
                  [[bitmap*systemColors.face_3d[2]]]]                  
  
  pState = Ptr_New({callback: callback, $
                    customColors: PTR_NEW(customColors), $
                    customIsList: customIsList, $
                    wCaller: wCaller, $
                    wTlb: wTlb, $
                    block: block, $
                    result: ptr_new(/ALLOCATE_HEAP), $
                    udata: ptr_new(udata), $
                    wColorSpace: wColorSpace, $
                    wSelected: wSelected, $
                    wLuminosity: wLuminosity, $
                    wPickRamp: wPickRamp, $
                    wOK: wOK, $
                    wR: wR, $
                    wG: wG, $
                    wB: wB, $
                    wH: wH, $
                    wS: wS, $
                    wL: wL, $
                    wTab: wTab, $
                    face3D: systemColors.Face_3D, $
                    bitmapPicker: bitmapPicker, $
                    currentVariable: '', $
                    colorSpaceBitmap: BytArr(xSizeColorMap, ySizeColorMap, 3), $
                    mouseDownColor: 0, $
                    mouseDownLuminosity: 0, $
                    moveEvents: KeyWord_Set(moveEvents)})
  Widget_Control, wBase, SET_UVALUE=pState  
  Widget_Control, wTab, SET_TAB_CURRENT=tabOnOpen
  Widget_Control, wBase, /REALIZE
  
  dialog_colorpickerUpdateColorSpace, wBase
  
  if (N_Elements(initialColor) eq 3) then begin
    Widget_Control, wR, SET_VALUE=StrTrim(Fix(initialColor[0]),2)
    Widget_Control, wG, SET_VALUE=StrTrim(Fix(initialColor[1]),2)
    Widget_Control, wB, SET_VALUE=StrTrim(Fix(initialColor[2]),2)
    dialog_colorpickerUpdateHLS, wBase
    dialog_colorpickerUpdateLuminosityRamp, wBase
  endif else begin  
    ; Let's select the red color in the corner
    sEvent = {WIDGET_DRAW, $
              ID: wColorSpace, $
              TOP: wBase, $
              HANDLER: wColorSpace, $
              TYPE: 1, $
              X: 0, $
              Y: ySizeColorMap-1, $
              PRESS: 1B, $
              RELEASE: 0B, $
              CLICKS: 1L, $
              MODIFIERS: 0L, $
              CH: 0B, $
              KEY: 0L}
    dialog_colorpickerColorSpaceEvent, sEvent
    sEvent = {WIDGET_DRAW, $
              ID: wColorSpace, $
              TOP: wBase, $
              HANDLER: wColorSpace, $
              TYPE: 1, $
              X: 0, $
              Y: ySizeColorMap-1, $
              PRESS: 0B, $
              RELEASE: 1B, $
              CLICKS: 1L, $
              MODIFIERS: 0L, $
              CH: 0B, $
              KEY: 0L}
    dialog_colorpickerColorSpaceEvent, sEvent
  endelse
  
  if (block) then begin
    Xmanager, 'dialog_colorpicker', wBase, NO_BLOCK=0, $
      EVENT_HANDLER='dialog_colorpickerEvent'
    result = *((*pState).result)
    if (ISA(result)) then begin
      customOut = result.customColors
      PTR_FREE, (*pState).result
      PTR_FREE, pState
      return, byte(result.value)
    endif else begin
      return, byte(0)
    endelse
  endif else begin
    Xmanager, 'dialog_colorpicker', wBase, /NO_BLOCK, $
      EVENT_HANDLER='dialog_colorpickerEvent'
    return, wBase
  endelse
  
end
