;
; Copyright (c) NV5 Geospatial Solutions, Inc. All rights reserved.
; Unauthorized reproduction prohibited.
;

;------------------------------------------------------------------------------
pro svg_icon_browser_test_icon, folder, icon, color, iconSize, $
  GROUP_LEADER=groupLeader
  compile_opt idl2

  image = Render_Icon(folder+':'+icon, $
                      WIDTH=iconSize, $
                      HEIGHT=iconSize, $
                      FOREGROUND_COLOR=color)

  wBase = Widget_Base(MBAR=mbar, /COL, TLB_FRAME_ATTR=1, $
    XSIZE=400, /MASK, BITMAP=image[*,*,0:2], TITLE='Testing icon: '+icon, $
    GROUP_LEADER=groupLeader)

  wm1 = Widget_Button(mbar, VALUE='File')
  !null = Widget_Button(wm1, VALUE='Open', IMAGE=image)
  !null = Widget_Button(wm1, VALUE='Save', IMAGE=image)
  
  wToolBar = Widget_Base(wBase, /TOOLBAR, /ROW)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)

  wFirst = Widget_Base(wBase, /ROW)

  wToolBar = Widget_Base(wFirst, /TOOLBAR, /ROW)
  !null = Widget_Button(wToolBar, VALUE=image)
  !null = Widget_Button(wToolBar, VALUE=image)
  !null = Widget_Button(wToolBar, VALUE=image)

  wSpace = Widget_Base(wFirst, XSIZE=15)

  wToolBar = Widget_Base(wFirst, /TOOLBAR, /ROW)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)

  wToolBar = Widget_Base(wFirst, /TOOLBAR, /ROW)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT, SENSITIVE=0)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT, SENSITIVE=0)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT, SENSITIVE=0)

  wSecond = Widget_Base(wBase, /ROW)

  wToolBar = Widget_Base(wSecond, /ROW)
  !null = Widget_Button(wToolBar, VALUE=image)
  !null = Widget_Button(wToolBar, VALUE=image)
  !null = Widget_Button(wToolBar, VALUE=image)

  wSpace = Widget_Base(wSecond, XSIZE=20)

  wToolBar = Widget_Base(wSecond, /ROW)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT)

  wToolBar = Widget_Base(wSecond, /ROW)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT, SENSITIVE=0)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT, SENSITIVE=0)
  !null = Widget_Button(wToolBar, VALUE=image, /FLAT, SENSITIVE=0)
  
  wTree = Widget_Tree(wBase, VALUE='', /FOLDER, BITMAP=image, /EXPANDED)
  wBranch1 = Widget_Tree(wTree, VALUE=icon, BITMAP=image, /FOLDER, /EXPANDED)
  wBranch2 = Widget_Tree(wTree, VALUE=icon, BITMAP=image, /FOLDER, /EXPANDED)  
  !null = Widget_Tree(wBranch1, VALUE=icon, BITMAP=image)
  !null = Widget_Tree(wBranch1, VALUE=icon, BITMAP=image)
  
  !null = Widget_Label(wBase, VALUE='Code to read icon:')
  str = "image = Render_Icon('"+folder+":"+icon+"', " + $
        "WIDTH="+iconSize.ToString()+", " + $
        "HEIGHT="+iconSize.ToString()+", " + $
        "FOREGROUND_COLOR=["+(color.ToString()).Join(',')+"])"
  wText = Widget_Text(wBase, $
                      VALUE=str, $
                      /WRAP, $
                      /SCROLL, $
                      YSIZE=4, $
                      ALIGN_CENTER=0, $
                      FONT='Courier*16')
  
  Widget_Control, wBase, /REALIZE
end

;------------------------------------------------------------------------------
pro svg_icon_browser_rebuild, wTop
  compile_opt idl2

  Widget_Control, wTop, GET_UVALUE=hState
  
  ; Remove any previous graphics.
  foreach oModel, hState['view'].get(/all) do begin
    if obj_valid(oModel) then obj_destroy, oModel
  endforeach
  
  Widget_Control, /HOURGLASS
  colors = Widget_Info(wTop, /SYSTEM_COLORS)
  
  Widget_Control, hState['folder'], GET_VALUE=folders
  index = Widget_Info(hState['folder'], /DROPLIST_SELECT)
  folder = (folders[index]).ToLower()
  subdir = ['resource','bitmaps','svg']
  if (index ne 0) then begin
    subdir = [subdir, folder]
  endif
  files = file_search(filepath('', subdir=subdir), '*.svg')

  Widget_Control, hState['size'], GET_VALUE=sizes
  index = Widget_Info(hState['size'], /DROPLIST_SELECT)
  iconSize = Fix(sizes[index])
  
  offset = 8
  iconArea = iconSize + offset
  
  hState['view'].GetProperty, DIMENSIONS=dims
  map = intarr(dims[0], dims[1]) - 1
  hState['names'] = List()
  hState['folders'] = List()
  
  Widget_Control, hState['color'], GET_UVALUE=fColor
  
  Widget_Control, hState['text'], GET_VALUE=filterText
  filterText = filterText.Trim()
  if (filterText ne '') then begin
    names = File_Basename(files)
    names = names.Replace('.svg', '')
    pos = Where(names.Contains(filterText, /FOLD_CASE) eq 1, /NULL)
    files = files[pos]
  endif
  
  cols = dims[0] / iconArea 
  i = -1
  for y=0, 9999 do begin
    for x=0, cols-1 do begin
      i++
      if (i ge N_Elements(files)) then begin
        ; We are done
        break
      endif
      if (dims[1]-y*iconArea-iconArea lt 0) then begin
        ; Icons too big, we run out of space
        break
      endif
      
      file = files[i]
      name = File_Basename(file)
      name = name.Replace('.svg', '')
      hState['names'].Add, name
      map[x*iconArea : x*iconArea+iconArea-1, $
          dims[1]-y*iconArea-iconArea : dims[1]-y*iconArea-1] = i 
      useFolder = (folder eq 'all') ? $
                    (StrSplit(File_Dirname(file), Path_Sep(), /EXTRACT))[-1] : $
                    folder
      hState['folders'].Add, useFolder      
      image = Render_Icon(useFolder+':'+name, $
                          WIDTH=iconSize, $
                          HEIGHT=iconSize, $
                          FOREGROUND_COLOR=fColor)
      oImage = Obj_New('IDLgrImage', $
                       image, $
                       INTERLEAVE=2, $
                       BLEND_FUNCTION=[3,4])
      oModel = Obj_New('IDLgrModel')      
      oModel.Translate, x*iconArea+offset/2, $
                        dims[1]-((y+1)*iconArea-offset/2), $
                        0
      oModel.Add, oImage
      hState['view'].Add, oModel
    endfor
  endfor
  
  hState['label_model'] = Obj_New('IDLgrModel')
  hState['label'] = Obj_New('IDLgrText', $
                            STRINGS='', $
                            COLOR=[0,0,255], $
                            /FILL_BACKGROUND, $
                            FILL_COLOR=[255,255,255])
  hState['label_model'].Add, hState['label']
  hState['view'].Add, hState['label_model']
  
  hState['map'] = map
  Widget_Control, hState['draw'], GET_VALUE=oWin 
  oWin.Draw
  Widget_Control, hState['draw'], SET_DRAW_VIEW=[0, dims[1]]
end

;------------------------------------------------------------------------------
pro svg_icon_browser_draw, ev
  compile_opt idl2

  stName = tag_names(ev, /STRUCTURE_NAME)
  Widget_Control, ev.top, GET_UVALUE=hState
  if (stName eq 'WIDGET_DRAW') then begin
    Widget_Control, hState['draw'], GET_VALUE=oWin
    oWin.SetCurrentCursor, 'ARROW' 
    case ev.type of 
      2: begin  ;move
        index = (hState['map'])[ev.x, ev.y]
        if (~obj_valid(hState['label'])) then begin
          Widget_Control, hState['draw'], GET_VALUE=oWin
          oWin.Draw
          return
        endif
        if (index ne -1) then begin
          hState['label_model'].SetProperty, HIDE=0
          name = (hState['names'])[index]
          hState['label'].SetProperty, STRINGS=' '+name+' '
          x = ev.x
          y = ev.y - 32
          hState['view'].GetProperty, DIMENSIONS=dims
          hState['label'].SetProperty, ALIGNMENT=(x gt dims[0]/2)
          hState['label_model'].SetProperty, TRANSFORM=[[1.0,0.0,0.0,x], $
                                                        [0.0,1.0,0.0,y], $
                                                        [0.0,0.0,1.0,0.0], $
                                                        [0.0,0.0,0.0,1.0]]
        endif else begin
          hState['label_model'].SetProperty, HIDE=1
        endelse
        Widget_Control, hState['draw'], GET_VALUE=oWin
        oWin.Draw
      end
      0: begin  ;click
        Widget_Control, hState['folder'], GET_VALUE=folders
        index = Widget_Info(hState['folder'], /DROPLIST_SELECT)
        index = (hState['map'])[ev.x, ev.y]
        if (index eq -1) then begin
          return
        endif
        name = (hState['names'])[index]
        folder = (hState['folders'])[index]
        Widget_Control, hState['color'], GET_UVALUE=color
        Widget_Control, hState['size'], GET_VALUE=sizes
        index = Widget_Info(hState['size'], /DROPLIST_SELECT)
        iconSize = Fix(sizes[index])
        svg_icon_browser_test_icon, folder, name, color, iconSize, GROUP_LEADER=ev.top
      end
      else: ;ignore      
    endcase 
  endif  
end

;------------------------------------------------------------------------------
pro svg_icon_browser_colorize_button, wButton, color
  compile_opt idl2

  bitmap = bytarr(16,16,3)
  bitmap[*,*,0] = color[0]
  bitmap[*,*,1] = color[1]
  bitmap[*,*,2] = color[2]
  Widget_Control, wButton, SET_VALUE=bitmap
end

;------------------------------------------------------------------------------
pro svg_icon_browser_event, ev
  compile_opt idl2

  case widget_info(ev.id, /UNAME) of
    'filter': begin
      svg_icon_browser_rebuild, ev.top
    end
    'folder': begin
      svg_icon_browser_rebuild, ev.top
    end
    'size': begin
      svg_icon_browser_rebuild, ev.top
    end
    'text': begin
      if (Tag_Names(ev, /STRUCTURE_NAME) eq 'WIDGET_TEXT_CH') then begin
        if (ev.ch eq 10b) then begin
          svg_icon_browser_rebuild, ev.top
        endif
      endif
    end
    'color': begin
      Widget_Control, ev.id, GET_UVALUE=color 
      newColor = Dialog_ColorPicker(DIALOG_PARENT=ev.id, $
                                    WCALLER=ev.id, $
                                    /MODAL, $
                                    INITIAL_COLOR=color)
      if (N_Elements(newColor) ne 3) then begin
        return
      endif
      Widget_Control, ev.id, SET_UVALUE=newColor
      svg_icon_browser_rebuild, ev.top
      svg_icon_browser_colorize_button, ev.id, newColor
    end    
    'top_base': begin
      ;Resize event
      Widget_Control, ev.top, GET_UVALUE=hState
      geom = Widget_Info(Widget_Info(hState['text'], /PARENT), /GEOMETRY)
      x = ev.x > 600
      y = ev.y > 400
      y = y - geom.scr_ySize      
      hState['view'].GetProperty, VIEWPLANE_RECT=v, $
                                  DIMENSIONS=d
      v[2] = x
      d[0] = x
      hState['view'].SetProperty, VIEWPLANE_RECT=v, $
                                  DIMENSIONS=d
      Widget_Control, hState['draw'], $
                      XSIZE=x, $
                      YSIZE=y, $
                      DRAW_XSIZE=x, $
                      SET_DRAW_VIEW=[0,9999]
      svg_icon_browser_rebuild, ev.top
    end
    else: ;ignore
  endcase
end

;------------------------------------------------------------------------------
pro svg_icon_browser
  compile_opt idl2

  totalDims = [920,16000]
  viewDims = [totalDims[0],600]

  tlb = Widget_Base(/COLUMN, $
                    TITLE='SVG Icon Browser', $
                    UNAME='top_base', $
                    /TLB_SIZE_EVENTS)
  colors = Widget_Info(tlb, /SYSTEM_COLORS)
  hState = Hash()
  
  hState['view'] = Obj_New('IDLgrView', $
                           COLOR=colors.face_3d, $
                           VIEWPLANE_RECT=[0,0,totalDims], $
                           DIMENSIONS=totalDims)

  row = Widget_Base(tlb, /ROW, /ALIGN_CENTER)
  
  !null = Widget_Label(row, VALUE='Category:')
  hState['folder'] = Widget_Droplist(row, $
                                     VALUE=['All','Solid','Regular','Brands'], $
                                     UNAME='folder')
  !null = Widget_Base(row, XSIZE=12)
  hState['text'] = Widget_Text(row, $
                               /EDITABLE, $
                               /ALL_EVENTS, $
                               UNAME='text')
  !null = Widget_Button(row, $
                        VALUE='Filter', $
                        UNAME='filter')
  !null = Widget_Base(row, XSIZE=12)
  !null = Widget_Label(row, VALUE='Color:')
  hState['color'] = Widget_Button(row, $
                                  /BITMAP, $
                                  UNAME='color', $
                                  UVALUE=colors.window_text)
  svg_icon_browser_colorize_button, hState['color'], colors.window_text 
  !null = Widget_Base(row, XSIZE=6)
  !null = Widget_Label(row, VALUE='Size:')
  hState['size'] = Widget_Droplist(row, $
                                   VALUE=['16','24','32','64'], $
                                   UNAME='size')
  Widget_Control, hState['size'], SET_DROPLIST_SELECT=2
  
  hState['draw'] = Widget_Draw(tlb, $
                               GRAPHICS_LEVEL=2, $
                               RETAIN=2, $
                               /MOTION_EVENTS, $
                               /BUTTON_EVENTS, $
                               X_SCROLL_SIZE=viewDims[0], $
                               Y_SCROLL_SIZE=viewDims[1], $
                               EVENT_PRO='svg_icon_browser_draw', $    
                               XSIZE=totalDims[0], $
                               YSIZE=totalDims[1], $
                               /SCROLL)

  row = Widget_Base(tlb, /ROW, /ALIGN_CENTER)
  !null = Widget_Label(row, VALUE='Click on an icon to test it in different widget types.')

  Widget_Control, tlb, /REALIZE
  Widget_Control, hState['draw'], GET_VALUE=oWin
  oWin.SetProperty, GRAPHICS_TREE=hState['view']
  Widget_Control, tlb, SET_UVALUE=hState

  svg_icon_browser_rebuild, tlb
  XManager, 'svg_icon_browser', tlb, /NO_BLOCK  
end