; Copyright (c) NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;+
; :Description:
;   This dialog allows the user to select a date and/or time from a calendar.
;
; :Category:
;   Compound widgets.
;
; :Params:
;
; :Keywords:
;
;-

;----------------------------------------------------------------------------
; Dialog_CalendarSafeDate
; Validate and catch common errors in a date string
pro Dialog_CalendarSafeDate, stringDate
  compile_opt idl2, hidden
  on_error, 2

  ; If empty, do nothing
  if strlen(stringDate) eq 0 then return
  
  ; Allow "today"
  if strtrim(strlowcase(stringDate), 2) eq 'today' then return

  ; Basic date format check: must be YYYY-MM-DD or YYYY-MM-DDTHH:MM:SS
  if STRMID(stringDate, 4, 1) ne '-' || STRMID(stringDate, 7, 1) ne '-' then $
    message, 'Invalid date string: expected "YYYY-MM-DD" or "YYYY-MM-DDTHH:MM:SS" format.'

  ColonPos = STRPOS(stringDate, ':')
  TPos = STRPOS(stringDate, 'T')

  ; If user includes full time with T separator → allow
  if (TPos ne -1) && (ColonPos ne -1) then return

  ; If user tries to include time but omits "T" separator → reject
  if (ColonPos ne -1) then $
    message, 'Invalid date-time string: missing "T" separator.'
  
end


;----------------------------------------------------------------------------
; dialog_calendarCleanup
;
pro dialog_calendarCleanup, wTop
  compile_opt idl2, hidden
  on_error, 2

  Widget_Control, wTop, GET_UVALUE = pState
  if Ptr_Valid(pState) then begin
    ; If it's blocked we handle this below.
    if ~(*pState).block then begin
      Ptr_Free, (*pState).udata
      Ptr_Free, pState
    endif
  endif
end

;----------------------------------------------------------------------------
; dialog_calendarGetAccumulatedWidgetOffset
;
function dialog_calendarGetAccumulatedWidgetOffset, wId, offset = offset, tlb = tlb
  compile_opt idl2, hidden
  on_error, 2

  if n_elements(offset) eq 0 then offset = [0, 0]
  if ~widget_info(wId, /valid_id) then return, 1

  geometry = widget_info(wId, /geometry)
  offset[0] += geometry.xoffset
  offset[1] += geometry.yoffset

  if ~widget_info(widget_info(wId, /parent), /valid_id) then begin
    tlb = wId
    offset[0] += (geometry.scr_xsize - geometry.xsize)
    offset[1] += (geometry.scr_ysize - geometry.ysize)
  endif

  return, dialog_calendarGetAccumulatedWidgetOffset(widget_info(wId, /parent), offset = offset, tlb = tlb)
end
;------------------------------------------------------------------------------
; Dialog_CalendarPickerRGBToHex
;
function dialog_calendarPickerRGBToHex, rgb
  compile_opt idl2, hidden
  on_error, 2

  if n_elements(rgb) ne 3 then return, !null

  result = ''
  for i = 0L, 2 do begin
    hexColor = strtrim(string(rgb[i], format = '(Z)'), 2)
    if strlen(hexColor) lt 2 then hexColor = '0' + hexColor
    result += hexColor
  endfor

  return, result
end

function dialog_calendarGetDefaultLocale
  compile_opt idl2, hidden
  on_error, 2
  ;Default case
  Flatpickr_Locale = "default"
  IDL_Locale = LOCALE_GET()

  ; if mac then we have UNIX language codes
  ; if windows then we have windows language codes
  ; handle
  if (!VERSION.OS_FAMILY eq "Windows") then begin

    linePos = STRPOS(IDL_Locale, "_")
    Locale = (linePos ge 0) ? strtrim(strmid(IDL_Locale, 0, linePos), 2) : IDL_Locale

    case Locale of
      'English':             Flatpickr_Locale = 'en'
      'French':              Flatpickr_Locale = 'fr'
      'German':              Flatpickr_Locale = 'de'
      'Spanish':             Flatpickr_Locale = 'es'
      'Italian':             Flatpickr_Locale = 'it'
      'Japanese':            Flatpickr_Locale = 'ja'
      'Korean':              Flatpickr_Locale = 'ko'
      'Dutch':               Flatpickr_Locale = 'nl'
      'Portuguese':          Flatpickr_Locale = 'pt'
      'Russian':             Flatpickr_Locale = 'ru'
      'Chinese':             Flatpickr_Locale = 'zh'
      'Arabic':              Flatpickr_Locale = 'ar'
      'Turkish':             Flatpickr_Locale = 'tr'
      'Swedish':             Flatpickr_Locale = 'sv'
      'Norwegian':           Flatpickr_Locale = 'no'
      'Danish':              Flatpickr_Locale = 'da'
      'Finnish':             Flatpickr_Locale = 'fi'
      'Polish':              Flatpickr_Locale = 'pl'
      'Czech':               Flatpickr_Locale = 'cs'
      'Hungarian':           Flatpickr_Locale = 'hu'
      'Hebrew':              Flatpickr_Locale = 'he'
      'Greek':               Flatpickr_Locale = 'gr'
      'Romanian':            Flatpickr_Locale = 'ro'
      'Slovak':              Flatpickr_Locale = 'sk'
      'Ukrainian':           Flatpickr_Locale = 'uk'
      'Vietnamese':          Flatpickr_Locale = 'vn'
      'Thai':                Flatpickr_Locale = 'th'
      'Indonesian':          Flatpickr_Locale = 'id'
      else: Flatpickr_Locale = "default"
    endcase

    ; handle subcases.
    dotPos = STRPOS(IDL_Locale, ".")
    Locale = (dotPos ge 0) ? strtrim(strmid(IDL_Locale, 0, dotPos), 2) : IDL_Locale

    case Locale of
      'Chinese_Traditional': Flatpickr_Locale = 'zh-tw'
      'Chinese_Taiwan': Flatpickr_Locale = 'zh-tw'
      'Serbian_Cyrillic': Flatpickr_Locale = 'sr-cyr'
      'Arabic_Algeria': Flatpickr_Locale = 'ar-dz'
      else: ;accept above FLatpickr_Locale
    endcase

  endif else begin

    linePos = STRPOS(IDL_Locale, "_")
    Flatpickr_Locale = (linePos ge 0) ? strtrim(strmid(IDL_Locale, 0, linePos), 2) : "default"

    ; handle subcases.
    dotPos = STRPOS(IDL_Locale, ".")
    Locale = (dotPos ge 0) ? strtrim(strmid(IDL_Locale, 0, dotPos), 2) : IDL_Locale

    case Locale of
      'zh_TW':      Flatpickr_Locale = 'zh-tw'
      'sr_RS':      Flatpickr_Locale = 'sr-cyr'
      'sr_Cyrl_RS': Flatpickr_Locale = 'sr-cyr'
      'ar_DZ':      Flatpickr_Locale = 'ar-dz'
      else:  ;accept above Flatpicker_Locale
    endcase
  endelse
  
  return, Flatpickr_Locale
end

pro dialog_calendar_callBackWithEvent, pState, sEvent, callBackEvent, DESTROY_WIDGET = destroyIn
  compile_opt idl2, hidden
  on_error, 2
  
  ; Extract the callback routine from pState
  routine = (*pState).callback

  ; Handle optional user data
  userdata = isa(*(*pState).udata) ? *(*pState).udata : !null
  
  if keyword_set(destroyIn) then begin
    widget_control, sEvent.top, /destroy
  endif

  ; If a callback routine is defined, invoke it
  if routine ne '' then begin
    ; Check if routine expects more than one parameter and userdata is valid
    if isa(userdata) and ((routine_info(routine, /parameters)).num_args gt 1) then begin
      call_procedure, routine, callBackEvent, userdata
      
      ; if we plan to keep using our pointers then we need to store UData for later
      if ~keyword_set(destroyIn) then begin
        *(*pState).udata = userdata
      endif
      
    endif else begin
      call_procedure, routine, callBackEvent
    endelse
  endif
end

;----------------------------------------------------------------------------
; Dialog_CalendarCleanString
function Dialog_CalendarCleanString, strOutput, listOutput, mode
  compile_opt idl2, hidden
  on_error, 2
  
  cleanedStr = strOutput
  
  if mode EQ 2 && listOutput.length GE 2 then begin
    toPos = strpos(strOutput, " to ")
    if toPos GE 0 then begin
      from = strtrim(strmid(strOutput, 0, toPos), 2)
      to   = strtrim(strmid(strOutput, toPos + 4), 2)
      cleanedStr = from + ' ... ' + to
    endif
  endif
  
  return, cleanedStr
end


;----------------------------------------------------------------------------
; Dialog_CalendarPickerBrowserEvent
;
pro dialog_calendarPickerBrowserEvent, sEvent
  compile_opt idl2, hidden
  on_error, 2
  
  widget_control, sEvent.top, get_uvalue = pState
  
  commandName = widget_info(sEvent.id, /uname)
  
  if TAG_NAMES(sEvent, /STRUCTURE_NAME) eq 'WIDGET_KILL_REQUEST' then begin
    commandName = "cancel"
  endif
  
  case commandName of

    'ok': begin
      if (*pState).currentValue eq '' then begin
        !null = dialog_message('Please select a date.', dialog_parent = sEvent.top)
        return
      endif
      
      ; Build the callback event structure
      ; we modify the original as to be nicer to the user
      callBackEvent = {id:sevent.id,$
      str_value: (*pState).currentValue,$
      value    : (*pState).currentValueList,$
      ok:    1b, $
      cancel: 0b}
      ; will destroy the widget.
      dialog_calendar_callBackWithEvent, pState, sEvent, callBackEvent, /DESTROY_WIDGET
      
    end

    'cancel': begin
            ; Start callback.
      (*pState).currentValue = ''
      (*pState).currentValueList = list()
      
      callBackEvent = {id:sevent.id,$
      str_value: (*pState).currentValue,$
      value    : (*pState).currentValueList,$
      ok:    0b, $
      cancel: 1b}
      
      dialog_calendar_callBackWithEvent, pState, sEvent, callBackEvent, /DESTROY_WIDGET
      
    end

    'browser': begin

      ; On pressing a calendar date.
      if sEvent.type eq 9 then begin
        hEvent = json_parse(sEvent.value)
        
        
        if hEvent.hasKey('DEBUG') then begin
          print, hEvent['DEBUG']
        endif
        
        if hEvent.hasKey('ERROR') then begin
          message, `${(hEvent['ERROR'])["message"]} ${(hEvent['ERROR'])["data"]}`, /INFORMATIONAL
        endif

        if hEvent.hasKey('resize') then begin
          eventinfo = hEvent['resize']
          widget_control, (*pState).wBase, xsize = ((eventinfo['width']) + (*pState).xpad), $
            ysize = ((eventinfo['height']) + (*pState).ypad), /realize
        endif

        if hEvent.hasKey('set') then begin
          
          hEvent = json_parse(sEvent.value)
          ; clean to -> ... in the case of mode = 2
          cleaned = Dialog_CalendarCleanString((hEvent["set"])["str"], (hEvent["set"])["list"], (*pState).mode)
          (*pState).currentValue = cleaned
          (*pState).currentValueList = (hEvent["set"])["list"]
          
          ; Start callback.
          callBackEvent = {id:sevent.id,$
            str_value: (*pState).currentValue,$
            value    : (*pState).currentValueList,$
            ok:    0b, $
            cancel: 0b}
          
          dialog_calendar_callBackWithEvent, pState, sEvent, callBackEvent
          
          widget_control, (*pState).wOK, sensitive = 1
        endif
      endif
    end

    else: ; ignore

  endcase
end

;----------------------------------------------------------------------------
; dialog_calendar
;
function dialog_calendar, $
ALT_INPUT       = altInputIn, $
ALT_EDITABLE = altAllowIn, $
CALLBACK        = callbackIn, $
DATE_FORMAT     = dateFormatIn, $
DIALOG_PARENT   = dialogParentIn, $
DISABLE_DATES   = disableIn, $
ENABLE_DATES    = enableIn, $
ENABLE_TIME     = enableTime, $
INITIAL_VALUE   = initialValue, $
LOCALE          = localein, $
MAX_DATE        = maxDateIn, $
MIN_DATE        = minDateIn, $
MODAL           = modalIn, $
MODE            = modein, $
NO_BLOCK        = blockIn, $
SCALE           = scaleIn, $
SELECTION_BUTTONS = selectionButtonsIn, $
TITLE           = titleIn, $
USER_DATA       = udata, $
WEEK_NUMBERS    = weekNumbersIn, $
XOFFSET         = xoffsetIn, $
YOFFSET         = yoffsetIn, $
$; Days of the week (kept together at the end)
DISABLE_SUNDAY          = sundayin, $
DISABLE_MONDAY          = mondayin, $
DISABLE_TUESDAY         = tuesdayin, $
DISABLE_WEDNESDAY       = wednesdayin, $
DISABLE_THURSDAY        = thursdayin, $
DISABLE_FRIDAY          = fridayin, $
DISABLE_SATURDAY        = saturdayin  
  compile_opt idl2
  on_error, 2

  isWindows = !version.os_family eq 'Windows'

  ; Start Keyword handling:
  
  ; Store tempURL to be added to the real URL later.
  toAddtoURL = ""
  
  
  ; ALT_INPUT and ALT_FORMAT
  if keyword_set(altInputIn) then begin
    if isa(altInputIn, /STRING) then begin
      altInputFormat = altInputIn[0]
      toAddtoURL += '&altFormat=' + altInputFormat
    endif
    
    toAddtoURL += '&altInput=' + "true"
    ; we move by -16 by default to make everything look nice. ALT_INPUT lives in that buffer so we
    ; need to tell the JS not to move anything.
    toAddtoURL += '&translateY=' + "0"
  endif

  ; ALT_EDITABLE
  ; Default is 1 but we are going to handle that in the JS.
  if keyword_set(altInputIn) && isa(altAllowIn) then begin
    if isa(altAllowIn, /number) then begin
      if (altAllowIn[0] ne 0) && (altAllowIn[0] ne 1) then message, 'Invalid value for ALT_EDITABLE.'
      
      ; only pass if its going to be 0 (false).
      if ~(altAllowIn[0]) then begin
        toAddtoURL += '&allowInput=' + `${altAllowIn[0]}`
      endif
    endif else message, 'Invalid value for ALT_EDITABLE.'
  endif
  
  ; BLOCK
  block = Keyword_Set(blockIn) ? 0b : 1b
  
  selectionButtons = (block && isa(selectionButtonsIn)) ? selectionButtonsIn : 1

  ; DIALOG_PARENT and MODAL
  ; Setting defaults to 0
  modal = 0b
  dialogParent = 0
  if KEYWORD_SET(dialogParentIn) then begin
    if ISA(dialogParentIn, /NUMBER) then begin
      dialogParent = dialogParentIn[0]

      ; Find top level base
      wTop = dialogParent
      while Widget_Info(Widget_Info(wTop, /PARENT), /VALID_ID) do begin
        wTop = Widget_Info(wTop, /PARENT)
      endwhile
      modal = Widget_Info(wTop, /MODAL) ? 1 : KEYWORD_SET(modalIn)
    endif else message, 'Invalid value for DIALOG_PARENT.'
  endif

  ; TITLE
  ; Setting defaults
  title = 'Select Date'
  if keyword_set(titleIn) then begin
    if isa(titleIn, /STRING) then begin
      title = titleIn[0]
    endif else message, 'Invalid value for TITLE.'
  endif

  ; CALLBACK
  ; Setting defaults
  callback = ''
  if keyword_set(callbackIn) then begin
    if isa(callbackIn, /STRING) then begin
      catch, err
      if err ne 0 then begin
        catch, /cancel
        message, 'Callback routine does not exist.'
      endif
      !NULL = routine_info(callbackIn[0], /PARAMETERS)
      callback = callbackIn[0]
      catch, /cancel
    endif else message, 'Invalid value for CALLBACK.'
  endif

  ; ENABLE_TIME
  strEnableTime = Keyword_Set(enableTime) ? 'true' : 'false'
  xSize = 320
  ySize = Keyword_Set(enableTime) ? 350 : 300
  
  ; DATE_FORMAT
  dateFormat = Keyword_Set(enableTime) ? 'Y-m-d\\TH:i:S' : 'Y-m-d'
  if keyword_set(dateFormatIn) then begin
    if isa(dateFormatIn, /STRING) then begin
      dateFormat = dateFormatIn[0]
    endif else message, 'Invalid value for DATE_FORMAT.'
  endif

  ; INITIAL_VALUE
  ; Setting defaults to !Null
  defaultDate = !NULL
  if keyword_set(initialValue) then begin
    if ~isa(initialValue, /STRING) then begin
      message, 'Invalid value for INITIAL_VALUE.'
    endif
    defaultDate = initialValue[0]
    toAddtoURL += '&defaultDate=' + defaultDate
  endif
  
  ;LOCALE
  ; allow in a string containing fltpkr language codes. We will dynamically load the language in JS.
  if keyword_set(localein) then begin
    if ~isa(localein,/STRING) then begin
      message, 'Invalid value for LOCALE.'
    endif
    toAddtoURL += `&locale=${localein[0]}`
  endif else begin
    cleanedLocale = dialog_calendarGetDefaultLocale()
    
    ; If we have "default" then we dont need to bother complicating the URL.
    if cleanedLocale ne "default" then begin
      toAddtoURL += `&locale=${cleanedLocale}`
    endif
  endelse
  
  ;MODE
  ; default is not adding to URL which results in the behavior of "single"
  mode = 0
  if keyword_set(modein) then begin
    if isa(modein, /NUMBER) then begin
      mode = fix(modein[0])
      
      case mode OF
      (0): begin
        toAddtoURL += '&mode=' + "single"
      end
      (1): begin
         toAddtoURL += '&mode=' + "multiple"
      end
      (2): begin
         toAddtoURL += '&mode=' + "range"
      end
      else: Message, 'Invalid value for MODE.'
      endcase
      
    endif else message, 'Invalid value for MODE.'
    
  endif
  
  ;SCALE
  ;default is 100% but 100% is 90% of real size.
  ; this means the default is 90% of original but we will normalize that to 100%
  scale = 0.9
  if keyword_set(scaleIn) then begin
    if ~isa(scaleIn, /NUMBER) then begin
      message, 'Invalid value for SCALE.'
    endif
    if scaleIn lt 0 then begin
      message, 'Invalid value for SCALE.'
    endif
    
    scale = scale * scaleIn
    toAddtoURL += `&scale=${scale}`
  endif
  xSize *= scale
  ySize *= scale

  ; XOFFSET and YOFFSET
  ; Setting defaults to 0
  wTlb = 0L
  xoffset = 0
  yoffset = 0

  if keyword_set(xoffsetIn) then begin
    if isa(xoffsetIn, /NUMBER) then xoffset = xoffsetIn[0] $
    else message, 'Invalid value for XOFFSET.'
  endif

  if keyword_set(yoffsetIn) then begin
    if isa(yoffsetIn, /NUMBER) then yoffset = yoffsetIn[0] $
    else message, 'Invalid value for YOFFSET.'
  endif
  
  ; DISABLE_DATES
  if keyword_set(disableIn) then begin
    disablePass = list()
  
    catch, err
    if err ne 0 then begin
      catch, /cancel
      message, 'Invalid value for DISABLE_DATES.'
    endif else begin
      if ~isa(disableIn, /scalar) then begin
        disableArr = isa(disableIn, "list") ? disableIn.toarray() : disableIn
      endif else begin
        disableArr = disableIn[0]
      endelse
    endelse
    catch, /cancel
  
    if ~isa(disableArr, /string) then begin
      message, 'Invalid value for DISABLE_DATES.'
    endif
  
    ; Extract ranges using ':'
    dotdotdotPos = STRPOS(disableArr, "...")
    toDate = list()
    fromDate = list()
    accum = list()
  
    i = -1
    foreach position, dotdotdotPos do begin
      i++
      if position eq -1 then begin   
        pass = disableArr[i]
        Dialog_CalendarSafeDate, pass
        accum.add, pass
      endif else begin
        from = strtrim(strmid(disableArr[i], 0, position), 2)
        to   = strtrim(strmid(disableArr[i], position + 3, strlen(disableArr[i])), 2)
        Dialog_CalendarSafeDate, from
        Dialog_CalendarSafeDate, to
        toDate.add, to
        fromDate.add, from
      endelse
    endforeach
  
    ; Add single dates
    disablePass += accum
  
    ; Add ranges
    accum = list()
    for i = 0, n_elements(fromDate) - 1 do begin
      ohash = orderedhash("from", fromDate[i], "to", toDate[i])
      accum.add, ohash
    endfor
  
    disablePass += accum
  
    ; Final URL append
    toAddtoURL += '&disable=' + JSON_SERIALIZE(disablePass)
  endif
   
   
  ; ENABLE_DATES
  if keyword_set(EnableIn) then begin
    enablePass = list()
  
    catch, err
    if err ne 0 then begin
      catch, /cancel
      message, 'Invalid value for ENABLE_DATES.'
    endif else begin
      if ~isa(EnableIn, /scalar) then begin
        EnableArr = isa(EnableIn, "list") ? EnableIn.toarray() : EnableIn
      endif else begin
        EnableArr = EnableIn[0]
      endelse
    endelse
    catch, /cancel
  
    if ~isa(EnableArr, /string) then begin
      message, 'Invalid value for ENABLE_DATES.'
    endif
  
    ; Extract ranges using ':'
    dotdotdotPos = STRPOS(EnableArr, "...")
    toDate = list()
    fromDate = list()
    accum = list()
  
    i = -1
    foreach position, dotdotdotPos do begin
      i++
      if position eq -1 then begin
        pass = EnableArr[i]
        Dialog_CalendarSafeDate, pass
        accum.add, pass
      endif else begin
        from = strtrim(strmid(EnableArr[i], 0, position), 2)
        to   = strtrim(strmid(EnableArr[i], position + 3, strlen(EnableArr[i])), 2)
        Dialog_CalendarSafeDate, from
        Dialog_CalendarSafeDate, to
        toDate.add, to
        fromDate.add, from
      endelse
    endforeach
  
    ; Add single dates
    enablePass += accum
  
    ; Add ranges
    accum = list()
    for i = 0, n_elements(fromDate) - 1 do begin
      ohash = orderedhash("from", fromDate[i], "to", toDate[i])
      accum.add, ohash
    endfor
  
    enablePass += accum
  
    ; Final URL append
    toAddtoURL += '&enable=' + JSON_SERIALIZE(enablePass)
  endif
   
 
  ;Handle Disable days IE:
  ; SUNDAY, MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY
  disabled_days = list()
  if keyword_set(sundayin) then disabled_days.add, 0
  if keyword_set(mondayin) then disabled_days.add, 1
  if keyword_set(tuesdayin) then disabled_days.add, 2
  if keyword_set(wednesdayin) then disabled_days.add, 3
  if keyword_set(thursdayin) then disabled_days.add, 4
  if keyword_set(fridayin) then disabled_days.add, 5
  if keyword_set(saturdayin) then disabled_days.add, 6

  if n_elements(disabled_days) gt 0 then begin
    toAddtoURL += '&disableWeekDays=' + JSON_SERIALIZE(disabled_days)
  endif
  
  ; Handle MIN_DATE.
  if keyword_set(minDateIn) then begin
    if isa(minDateIn, /STRING) then begin
      minDate = minDateIn[0]
      Dialog_CalendarSafeDate, minDate
      toAddtoURL += '&minDate=' + minDateIn
    endif else message, 'Invalid value for MIN_DATE.'
  endif

  ; Handle MAX_DATE.
  if keyword_set(maxDateIn) then begin
    if isa(maxDateIn, /STRING) then begin
      maxDate = maxDateIn[0]
      Dialog_CalendarSafeDate, maxDate
      toAddtoURL += '&maxDate=' + maxDate
    endif else message, 'Invalid value for MAX_DATE.'
  endif
  
  ; WEEK_NUMBERS
  if keyword_set(weekNumbersIn) then begin
    toAddtoURL += '&weekNumbers=' + "true"
  endif
  
  ;add local
  groupLeader = dialogParent

  if isa(dialogParent) && widget_info(dialogParent, /VALID_ID) then begin
    offset = [0, 0]
    offsetAdjust = isWindows ? 6 : 0
    !null = dialog_calendarGetAccumulatedWidgetOffset(dialogParent, OFFSET = offset, TLB = wTlb)
    geom = Widget_Info(dialogParent, /GEOMETRY)
    xoffset += offset[0] - offsetAdjust
    yoffset += offset[1] + geom.SCR_YSIZE - offsetAdjust
  endif

  ; Start widget setup
  if groupLeader gt 0 then begin
    if ~isWindows && Widget_Info(groupLeader, /TYPE) ne 0 then begin
      groupLeader = Widget_Info(groupLeader, /PARENT)
    endif
  endif
  
  wBase = Widget_Base(XPAD = 0, YPAD = 0, SPACE = 0, $
    TITLE = title, /COLUMN, UNAME = 'tlb', MODAL = modal, $
    GROUP_LEADER = groupLeader, TLB_FRAME_ATTR = 1, $
    XOFFSET = xOffset, YOFFSET = yOffset, $
    KILL_NOTIFY = 'dialog_calendarCleanup', /TLB_KILL_REQUEST_EVENTS)
  
  ; Pick our colors then link it to JS
  systemColors = Widget_Info(wBase, /SYSTEM_COLORS)
  css = (!theme eq 'Dark') ? 'dark.css' : 'light.css'
  bgColor = Dialog_CalendarPickerRGBToHex(systemColors.face_3d)

  url = 'file://' + FILEPATH('index.htm', SUBDIRECTORY=['resource','webapps','dialog_calendar'])

  url += '?enableTime=' + strEnableTime + $
    '&dateFormat=' + dateFormat + $
    '&css=' + css + $
    '&bgColor=' + bgColor
  
  ; add most of the data we got from keywords. 
  url += toAddtoURL
  
  ; real quick set up padding.
  ; 8 pad for the  px margine defined in index.htm
  xpad = 8
  ypad = 8

  ; round up 2 and 4 are arbitrary.
  buttonBaseyPad = round((4 * scale) + 0.5)

  ypad += buttonBaseyPad
  
  wBrowser = Widget_Browser(wBase, $
    VALUE = url, UNAME = 'browser', $
    EVENT_PRO = 'Dialog_CalendarPickerBrowserEvent', $
    XSIZE = xSize, YSIZE = ySize)
  
  if (selectionButtons) then begin
    wOKBase = Widget_Base(wBase, $
      xpad = 2, ypad = buttonBaseyPad, SPACE = 0, $
      /ROW, /ALIGN_RIGHT, /GRID_LAYOUT)
    
    wOK = Widget_Button(wOKBase, VALUE = 'OK', UNAME = 'ok', $
      SENSITIVE = (N_Elements(initialValue) ne 0), $
      ACCELERATOR = 'return')
  
    wCancel = Widget_Button(wOKBase, VALUE = 'Cancel', $
      UNAME = 'cancel', ACCELERATOR = 'Escape')
      
    geom = WIDGET_INFO(wOKBase, /GEOMETRY)
    ypad +=  geom.YSIZE
  endif else begin
    wOK = 0
  endelse
  
  pState = Ptr_New({callback: callback, $
    dialogParent: dialogParent, wBase: wBase, block: block, $
    udata: ptr_new(udata), wBrowser: wBrowser, $
    currentValue: ((N_Elements(initialValue) eq 0) ? "" : initialValue), $
    currentValueList: ((N_Elements(initialValue) eq 0) ? list() : list(initialValue)), $
    wOK: wOK, XPAD: xpad, YPAD: ypad, Mode: mode})

  Widget_Control, wBase, SET_UVALUE = pState
  Widget_Control, wBase, /REALIZE
  ;end set up widget.
  
  ; Logic loop
  if block then begin
    Xmanager, 'Dialog_CalendarPicker', wBase, $
      NO_BLOCK = 0, EVENT_HANDLER = 'Dialog_CalendarPickerBrowserEvent'
    result = (*pState).currentValue
    
    udata = ISA(*(*pState).udata) ? *(*pState).udata : !NULL
  endif else begin
    
    Xmanager, 'Dialog_CalendarPicker', wBase, $
      /NO_BLOCK, EVENT_HANDLER = 'Dialog_CalendarPickerBrowserEvent'
    result = wBase
  endelse
  return, result
end
