;-----------------------------------------------------------------------------
function HttpRequest::Init, _INTERNAL=_internal
  compile_opt idl2, hidden, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo
  @httprequest_options
  if (~keyword_set(_internal)) then begin
    message, 'To create an HttpRequest object, call a ' + $
      'static method such as Get, Post, Put, or Delete.'
  endif
  self.curl = httprequest_init()
  self.slists = list()
  return, 1
end


;-----------------------------------------------------------------------------
pro HttpRequest::Cleanup
  compile_opt idl2, hidden, nosave
  on_error, 2
  foreach slist, self.slists do begin
    httprequest_slist_freeall, slist
  endforeach
  httprequest_mime_free, self.mime
  httprequest_cleanup, self.curl
end


;-----------------------------------------------------------------------------
pro HttpRequest::GetProperty, $
  CONTENT=content, $
  CURL=curl, $
  HEADERS=headers, $
  OK=ok, $
  STATUS_CODE=statusCode, $
  TEXT=text, $
  URL=url, $
  _REF_EXTRA=extra

  compile_opt idl2, hidden, nosave
  common httprequestoptions, httpopt, httpinfo
  on_error, 2

  if (~ptr_valid(self.pdata)) then begin
    self.pdata = ptr_new(0b)
  endif
  if (~ptr_valid(self.pheaders)) then begin
    self.pheaders = ptr_new(0b)
  endif

  if arg_present(content) then begin
    content = (self.statusCode ge 100) ? *self.pdata : ''
  endif

  if arg_present(curl) then begin
    curl = self.curl
  endif

  if arg_present(headers) then begin
    h = string(*self.pheaders)
    h = h.split(`\r\n`)
    h = h[where(h ne '', /null)]
    headers = OrderedHash()
    foreach h1, h do begin
      pieces = h1.split(':')
      if (pieces.length ge 2 && pieces[0] ne '') then begin
        headers[pieces[0]] = ((pieces[1:*]).join(':')).trim()
      endif
    endforeach
  endif

  if arg_present(ok) then begin
    ok = (self.statusCode ge 100 && self.statusCode lt 400) ? !true : !false
  endif

  if arg_present(statusCode) then begin
    statusCode = self.statusCode
  endif

  if arg_present(text) then begin
    text = string(*self.pdata)
  endif

  if arg_present(url) then begin
    url = self.url
  endif

  if (isa(extra)) then begin
    props = ['DIM', 'LENGTH', 'NDIM', 'TNAME', 'TYPECODE', 'TYPENAME', 'TYPESIZE']
    self.IDL_Object::GetProperty, _EXTRA=props
    tags = tag_names(HTTPINFO)
    foreach key, extra do begin
      if (max(props eq key)) then continue
      match = (where(tags eq key))[0]
      ; Always prefer the _T version because it is more precise.
      match_t = (where(tags eq (key + '_T')))[0]
      match = (match_t ge 0) ? match_t : match
      if (match ge 0) then begin
        result = httprequest_getinfo(self.curl, httpinfo.(match))
        (scope_varfetch(key, /ref_extra)) = result
      endif else begin
        message, `Unknown property '${key}'`
      endelse
    endforeach
  endif
end


;-----------------------------------------------------------------------------
function HttpRequest::JSON, QUIET=quiet, _EXTRA=_extra
  compile_opt idl2, hidden, nosave
  on_error, 2
  if (keyword_set(quiet)) then begin
    catch, err
    if (err ne 0) then begin
      catch, /cancel
      return, ''
    endif
  endif
  json = json_parse(*self.pdata, _EXTRA=_extra)
  return, json
end


;-----------------------------------------------------------------------------
function HttpRequest::Escape, data
  compile_opt idl2, hidden, static, nosave
  on_error, 2
  result = httprequest_escape(data)
  return, result
end


;-----------------------------------------------------------------------------
function HttpRequest::Unescape, data, BYTE=byte
  compile_opt idl2, hidden, static, nosave
  on_error, 2
  result = httprequest_unescape(data, keyword_set(byte))
  return, result
end

;-----------------------------------------------------------------------------
function HttpRequest::_combineParams, params, ESCAPE=escape
  compile_opt idl2, hidden, nosave
  on_error, 2

  if (isa(params, 'hash')) then begin
    payload = strarr(params.length)
    i = 0
    foreach value, params, key do begin
      if (~isa(value, /string)) then begin
        value = (string(value)).trim()
      endif
      if (keyword_set(escape)) then begin
        key = HttpRequest.escape(key)
        value = HttpRequest.escape(value)
      endif
      payload[i] = key + '=' + value
      i++
    endforeach
    payload = payload.join('&')
  endif else if (isa(params, /string)) then begin
    payload = params.join('&')
  endif else begin
    message, 'PARAMS must be a hash, string array, or scalar string', level = -1
  endelse
  return, payload
end


;-----------------------------------------------------------------------------
function HttpRequest::_constructHeaders, reqheaders, slist
  compile_opt idl2, hidden, nosave
  on_error, 2

  if (~isa(slist)) then slist = 0LL
  if (isa(reqheaders, 'hash') || isa(reqheaders, 'struct')) then begin
    headers = isa(reqheaders, 'struct') ? hash(reqheaders) : reqheaders
    foreach value, headers, key do begin
      key1 = key.endsWith(':') ? key : key + ':'
      isEmptyHeader = key.endsWith(';')
      if (~isEmptyHeader && ~isa(value)) then begin
        message, `Undefined header value for header key: '${key}'`
      endif
      keyvalue = isEmptyHeader ? key : key1 + value
      slist = httprequest_slist_append(slist, keyvalue)
    endforeach
  endif else begin
    message, 'HEADERS must be a hash, dictionary, or structure.', level = -1
  endelse
  return, slist
end


;-----------------------------------------------------------------------------
pro HttpRequest::_handleHeaders, reqheaders
  compile_opt idl2, hidden, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo
  if (isa(reqheaders)) then begin
    self.reqheaders = self._constructHeaders(reqheaders, self.reqheaders)
  endif
  if (self.reqheaders ne 0) then begin
    ; Mark for future deletion
    self.slists.Add, self.reqheaders
    !null = httprequest_setopt(self.curl, httpopt.HTTPHEADER, self.reqheaders)
  endif
end


;-----------------------------------------------------------------------------
; Special case for curl_slist options
;
pro HttpRequest::_handleSlistOption, option, value
  compile_opt idl2, hidden, nosave
  on_error, 2
  slist = 0
  foreach v, value do begin
    slist = httprequest_slist_append(slist, v)
  endforeach
  !null = httprequest_setopt(self.curl, option, slist)
  ; Mark for future deletion
  self.slists.Add, slist
end


;-----------------------------------------------------------------------------
pro HttpRequest::_handleOptions, optionsIn
  compile_opt idl2, hidden, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo

  tags = tag_names(HTTPOPT)
  if (isa(optionsIn, 'hash') || isa(optionsIn, 'struct')) then begin
    options = isa(optionsIn, 'struct') ? hash(optionsIn) : optionsIn
    foreach value, options, key do begin
      match = where(tags eq key.toUpper(), /null)
      if (isa(match)) then begin
        ; Special case for some weird curl_slist options
        if (httpopt.(match) eq httpopt.resolve || $
          httpopt.(match) eq httpopt.httpheader || $
          httpopt.(match) eq httpopt.proxyheader) then begin
          self._handleSlistOption, httpopt.(match), value
        endif else begin
          !null = httprequest_setopt(self.curl, httpopt.(match), value)
        endelse
      endif else begin
        message, `Unknown option: '${key.toUpper()}', skipping...`, /info, level = -1
      endelse
    endforeach
  endif else begin
    message, 'OPTIONS must be a hash, dictionary, or structure.', level = -1
  endelse
end


;-----------------------------------------------------------------------------
pro HttpRequest::_setDefaultOptions
  compile_opt idl2, hidden, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo
  !null = httprequest_setopt(self.curl, httpopt.AUTOREFERER, 1)
  !null = httprequest_setopt(self.curl, httpopt.FOLLOWLOCATION, 1)
  !null = httprequest_setopt(self.curl, httpopt.HTTPAUTH, 0x7FFFFFEF)
  !null = httprequest_setopt(self.curl, httpopt.PROXYAUTH, 0x7FFFFFEF)
end

;-----------------------------------------------------------------------------
; Intercept call back for progress_bar keyword
function HttpRequest_ProgressCB, downTotal, downNow, upTotal, upNow, callbackData
  compile_opt idl2, hidden, nosave
  on_error, 2
  common httpRequestProgressCommon, _origCallBack

  percent = round(100 * float(downNow) / (float(downTotal) > 1))
  
  ; we always want to return either 1 or the value of the of the callBack.
  ; Returning a 0 will abort the Curl Call. 
  result = 1
  if isa(_origCallBack) then begin
    result = call_function(_origCallBack,  downTotal, downNow, upTotal, upNow, callbackData)
  endif
  
  ; Update progress bar after call_function in case something goes wrong with call_function
  ; (IE: invalid call back name). This prevents a empty progress bar from appearing before mssg.

   cli_progress.update, percent
   if result eq 0 then cli_progress.newline ; if result eq 0 then abandon the progress bar.
  
  return, result
end

;-----------------------------------------------------------------------------
function HttpRequest::Get, urlIn, $
  ESCAPE=escape, $
  HEADERS=reqheaders, $
  OPTIONS=options, $
  PARAMS=params, $
  CALLBACK_FUNCTION=callbackFunc, $
  CALLBACK_DATA=callbackData, $
  PROGRESS_BAR=progress
  compile_opt idl2, hidden, static, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo
  obj = HttpRequest(/_internal)
  self = obj
  
  ; If we want a progress bar figure out what we want to store for later.
  ; Then slip in HttpRequest_ProgressCB to intercept the call back and call cli_progress
  ; We are using a common block here because the cli_progress is only capable of
  ; printing one bar at a time and thus we dont gain much from trying to keep track
  ; of which loading bar is which
  if (keyword_set(progress)) then begin
    common httpRequestProgressCommon, _origCallBack
    _origCallBack = !NULL
    cli_progress.INITIALIZE, preset = "retro", spinner = 0, /REMAINING, AUTO_FINISH = !FALSE
    
    if (isa(callbackFunc)) then begin
      _origCallBack = callbackFunc
    endif
    
    callbackFunc = "HttpRequest_ProgressCB"
  endif

  self.url = urlIn[0]
  if (isa(params)) then begin
    self.url += '?' + self._combineParams(params, ESCAPE=escape)
  endif

  os = (!version.os eq 'Win32') ? '' : (!version.os + '.')
  cainfo = `${!dir}/bin/bin.${os}${!version.arch}/ca-bundle.crt`
  !null = httprequest_setopt(self.curl, httpopt.cainfo, cainfo)
  !null = httprequest_setopt(self.curl, httpopt.url, self.url)

  self._handleHeaders, reqheaders

  self._setDefaultOptions

  if (isa(options)) then begin
    self._handleOptions, options
  endif

  self.statusCode = httprequest_perform(self.curl, dataOut, headersOut, callbackFunc, callbackData)

  self.url = httprequest_getinfo(self.curl, httpinfo.EFFECTIVE_URL)
  self.pdata = ptr_new(dataOut, /no_copy)
  self.pheaders = ptr_new(headersOut, /no_copy)
  ; If COOKIEJAR was specified, output any cookies now.
  !null = httprequest_setopt(self.curl, httpopt.COOKIELIST, "FLUSH")
  if (keyword_set(progress)) then cli_progress.finish
  return, obj
end


;-----------------------------------------------------------------------------
pro HttpRequest::_postPayload, payload
  compile_opt idl2, hidden, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo
  ; We must send the POSTFIELDSIZE first, so COPYPOSTFIELDS knows
  ; how much data to copy.
  len = isa(payload, /string, /scalar) ? strlen(payload) : payload.length
  !null = httprequest_setopt(self.curl, httpopt.POSTFIELDSIZE, len)
  ; Use COPYPOSTFIELDS, not POSTFIELDS, so it makes a copy of our temporary
  ; string variable. Otherwise the string gets freed before it can be sent.
  !null = httprequest_setopt(self.curl, httpopt.COPYPOSTFIELDS, payload)
end


;-----------------------------------------------------------------------------
pro HttpRequest::_handlePostJSON, json
  compile_opt idl2, hidden, nosave
  on_error, 2
  self.reqheaders = httprequest_slist_append(self.reqheaders, 'Content-Type: application/json')
  self._postPayload, isa(json, /STRING, /SCALAR) ? json : json_serialize(json)
end


;-----------------------------------------------------------------------------
pro HttpRequest::_handlePostMultipart, data
  compile_opt idl2, hidden, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo
  if (~isa(data, 'hash')) then begin
    message, 'Multipart must be a hash or dictionary.'
  endif
  if (~self.mime) then self.mime = httprequest_mime_init(self.curl)
  foreach value, data, key do begin
    part = httprequest_mime_addpart(self.mime)
    !null = httprequest_mime_name(part, key)
    if (isa(value, /string) || isa(value, 'byte')) then begin
      !null = httprequest_mime_data(part, value)
    endif else if (isa(value, 'hash')) then begin
      !null = httprequest_mime_data(part, json_serialize(value))
    endif else if (isa(value, 'struct')) then begin
      val = isa(value, 'struct') ? hash(value) : value
      foreach field, val, fname do begin
        case strlowcase(fname) of
          'value': !null = httprequest_mime_data(part, $
            isa(value, 'hash') ? json_serialize(field) : field)
          'file': !null = httprequest_mime_filedata(part, field)
          'mimetype': !null = httprequest_mime_type(part, field)
          'headers': !null = httprequest_mime_headers(part, self._constructHeaders(field))
          else: message, `Unknown structure field '${fname}' for multipart value '${key}'`
        endcase
      endforeach
    endif else begin
      message, 'Multipart value must be a string, byte array, structure, or hash.'
    endelse
  endforeach
  !null = httprequest_setopt(self.curl, httpopt.MIMEPOST, self.mime)
end


;-----------------------------------------------------------------------------
pro HttpRequest::_doPost, urlIn, $
  DATA=data, $
  ESCAPE=escape, $
  HEADERS=reqheaders, $
  JSON=json, $
  MULTIPART=multipart, $
  OPTIONS=options, $
  PARAMS=params, $
  CALLBACK_FUNCTION=callbackFunc, $
  CALLBACK_DATA=callbackData, $
  PROGRESS_BAR=progress

  compile_opt idl2, hidden, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo

  ; If we want a progress bar figure out what we want to store for later.
  ; Then slip in HttpRequest_ProgressCB to intercept the call back and call cli_progress
  ; We are using a common block here because the cli_progress is only capable of
  ; printing one bar at a time and thus we dont gain much from trying to keep track
  ; of which loading bar is which
  if (keyword_set(progress)) then begin
    common httpRequestProgressCommon, _origCallBack
    _origCallBack = !NULL
    CLI_PROGRESS.INITIALIZE, preset = "retro", spinner = 0, /REMAINING, AUTO_FINISH = !FALSE

    if (isa(callbackFunc)) then begin
      _origCallBack = callbackFunc
    endif

    callbackFunc = "HttpRequest_ProgressCB"
  endif

  self.url = urlIn[0]
  os = (!version.os eq 'Win32') ? '' : (!version.os + '.')
  cainfo = `${!dir}/bin/bin.${os}${!version.arch}/ca-bundle.crt`
  !null = httprequest_setopt(self.curl, httpopt.cainfo, cainfo)
  !null = httprequest_setopt(self.curl, httpopt.url, self.url)
  ; Start with a size of 0 to prevent hangs on Unix, which will otherwise wait for data.
  !null = httprequest_setopt(self.curl, httpopt.POSTFIELDSIZE, 0)

  if (isa(data)) then begin
    if (~isa(data, 'byte')) then begin
      message, 'Unknown data type'
    endif
    ; If the user hasn't specified Content-Type, then default to application/octet-stream
    hasContentType = 0
    if (isa(reqheaders)) then begin
      keys = isa(reqheaders, 'struct') ? tag_names(reqheaders) : (reqheaders.keys()).toArray()
      hasContentType = max(strupcase(keys) eq 'CONTENT-TYPE') eq 1
    endif
    if (~hasContentType) then begin
      self.reqheaders = httprequest_slist_append(self.reqheaders, 'Content-Type: application/octet-stream')
    endif
    self._postPayload, data
  endif

  if (isa(params)) then begin
    payload = self._combineParams(params, ESCAPE=escape)
    self._postPayload, payload
  endif

  if (isa(multipart)) then begin
    self._handlePostMultipart, multipart
  endif

  if (isa(json)) then begin
    self._handlePostJSON, json
  endif

  ; By default turn off the "Expect: 100-continue" header.
  ; Most POST/PUT bodies are small, and "Expect: 100-continue" introduces
  ; complexities for the server and problems for proxies.
  self.reqheaders = httprequest_slist_append(self.reqheaders, 'Expect:')

  self._handleHeaders, reqheaders

  self._setDefaultOptions

  if (isa(options)) then begin
    self._handleOptions, options
  endif

  self.statusCode = httprequest_perform(self.curl, dataOut, headersOut, callbackFunc, callbackData)

  self.url = httprequest_getinfo(self.curl, httpinfo.effective_url)
  self.pdata = ptr_new(dataOut, /no_copy)
  self.pheaders = ptr_new(headersOut, /no_copy)
  ; If COOKIEJAR was specified, output any cookies now.
  !null = httprequest_setopt(self.curl, httpopt.COOKIELIST, "FLUSH")
  
  if (keyword_set(progress)) then cli_progress.finish
end


;-----------------------------------------------------------------------------
function HttpRequest::Post, url, _REF_EXTRA=extra

  compile_opt idl2, hidden, static, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo

  obj = HttpRequest(/_internal)
  self = obj
  !null = httprequest_setopt(self.curl, httpopt.POST, !true)
  self._doPost, url, _EXTRA=extra
  return, obj
end


;-----------------------------------------------------------------------------
function HttpRequest::Put, url, _REF_EXTRA=extra

  compile_opt idl2, hidden, static, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo

  obj = HttpRequest(/_internal)
  self = obj

  ; Libcurl docs say to set httpopt.UPLOAD for PUT, but it doesn't work.
  ; I had curl generate sample code using:
  ;   curl -X PUT http://localhost:3000 -H "Content-Type: application/json"
  ;     -d '{"key":"value"}' --libcurl foo.c
  ; The sample code used httpopt.CUSTOMREQUEST instead of httpopt.UPLOAD,
  ; which seems to work properly.
  !null = httprequest_setopt(self.curl, httpopt.CUSTOMREQUEST, "PUT")

  self._doPost, url, _EXTRA=extra
  return, obj
end


;-----------------------------------------------------------------------------
function HttpRequest::Delete, urlIn, _REF_EXTRA=extra

  compile_opt idl2, hidden, static, nosave
  on_error, 2
  common httprequestoptions, httpopt, httpinfo

  obj = HttpRequest(/_internal)
  self = obj
  !null = httprequest_setopt(self.curl, httpopt.CUSTOMREQUEST, "DELETE")
  self.url = urlIn[0]
  self._doPost, self.url, _EXTRA=extra
  return, obj
end


;-----------------------------------------------------------------------------
pro HTTPRequest__define
  compile_opt idl2, hidden, nosave
  !null = {HTTPRequest, $
    inherits IDL_Object, $
    curl: 0LL, $
    reqheaders: 0LL, $
    mime: 0LL, $
    statusCode: 0L, $
    progress: '', $
    url: '', $
    slists: list(), $
    pdata: ptr_new(), $
    pheaders: ptr_new() $
  }
end
