; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;

;----------------------------------------------------------------------------
;+
; :Returns:
;   a string giving the filename of the external file
;
; :Arguments:
;   filename: in, required, String
;     a string giving the filename
;   block: in, required, Structure<any>
;     a raw data block structure
;   nexternal: in, required, Integer
;     a running counter used for filenames for the external data blocks
;
;-
function asdf_write_external_block, filename, block, nexternal
  compile_opt idl2, hidden
  asdf = ASDF_File()
  externalfile = filename
  if (externalfile.endsWith('.asdf', /fold_case)) then begin
    externalfile = externalfile.remove(-5)
  endif
  externalfile += `${nexternal,'%04d'}.asdf`
  openw, lun, externalfile, /get_lun
  printf, lun, asdf
  printf, lun, '...'
  writeu, lun, block
  free_lun, lun, /force
  return, file_basename(externalfile)
end

;----------------------------------------------------------------------------
;+
; :Description:
;   Add an internal data block to our cache (to be written later),
;   or write out an external data block
;
; :Arguments:
;   item: in, required, ASDF_NDArray
;     an ASDF_NDArray item to output to the file
;   filename: in, required, String
;     a string giving the filename
;   blocks: bi, required, Structure<any>
;     a list of data block structures
;   nexternal: bi, required, Integer
;     a running counter used for filenames for the external data blocks
;
;-
pro asdf_write_handleblock, item, filename, blocks, nexternal
  compile_opt idl2, hidden

  ; Inline array?
  if (~item.hasKey('source')) then return
  source = item.source
  block = item.getBlock()

  if (isa(source, /number)) then begin
    if (~isa(blocks)) then blocks = list()
    item.source = blocks.length
    blocks.add, block
    return
  endif

  if (isa(source, /string)) then begin
    ; Start our external filename counter, for files 0000, 0001, 0002,...
    if (~isa(nexternal)) then nexternal = 0LL
    item.source = asdf_write_external_block(filename, block, nexternal)
    nexternal++
  endif
end

;----------------------------------------------------------------------------
;+
; :Description:
;   Retrieve our binary data blocks as a list of block structures.
;   Writes out any external data blocks, so we don't have to cache the data.
;
; :Arguments:
;   node: in, required, Hash<any> | List<any>
;     a list or hash containing (possibly) some ASDF_NDArray nodes
;   filename: in, required, String
;     a string giving the filename
;   blocks: bi, required, List<Structure<any>>
;     a list of data block structures
;   nexternal: in, required, Integer
;     a running counter used for filenames for the external data blocks
;
;-
pro asdf_write_walk_metadata, node, filename, blocks, nexternal
  compile_opt idl2, hidden

  if (isa(node, 'list') || isa(node, 'hash')) then begin
    foreach item, node, key do begin
      if (isa(item, 'ASDF_NDArray')) then begin
        asdf_write_handleblock, item, filename, blocks, nexternal
      endif else if (isa(item, /scalar) && isa(item, /complex)) then begin
        node[key] = yaml_value(item, tag='!core/complex-1.0.0')
      endif else begin
        asdf_write_walk_metadata, item, filename, blocks, nexternal
      endelse
    endforeach
  endif
end

;----------------------------------------------------------------------------
;+
; :Description:
;   Write out all of our internal data blocks to the end of our file.
;
; :Arguments:
;   filename: in, required, String
;     a string giving the filename
;   blocks: in, required, List<Structure<any>>
;     a list of data block structures, in raw form ready to be written
;
;-
pro asdf_write_append_internal_blocks, filename, blocks
  compile_opt idl2, hidden
  openw, lun, filename, /get_lun, /append
  foreach block, blocks do begin
    writeu, lun, block
  endforeach
  free_lun, lun, /force
end

;----------------------------------------------------------------------------
;+
; :Description:
;   The asdf_write function takes an ASDF_File object and
;   writes out an ASDF file.
;
; :Arguments:
;   filename: in, required, String
;     A string giving the full filename where the data should be written.
;   asdf: in, required, ASDF_File
;     *Asdf* must be an ASDF_File object containing the data.
;
; :Keywords:
;   debug: in, optional, Boolean
;     Set this keyword to disable error catching. Undocumented.
;
; :Author:
;   CT, March 2023.
;
;-
pro asdf_write, filename, asdf, DEBUG=debug

  compile_opt idl2, hidden
  lun = 0
  if (~keyword_set(debug)) then begin
    on_error, 2
    catch, ierr
    if (ierr ne 0) then begin
      catch, /cancel
      if (lun ne 0) then free_lun, lun, /force
      msg = (!error_state.msg).Replace('YAML_PARSE_INTERNAL: ', '')
      message, msg
    endif
  endif

  catch, err
  if (err eq 0) then begin
    idlsoftware = YAML_Map('name', 'IDL', 'version', `${!version.release}`)
    idlsoftware.tag = '!core/software-1.0.0'
    idlsoftware.style = 'flow'
    asdf['history', 'extensions', 0, 'software'] = idlsoftware
  endif
  catch, /cancel

  asdf_write_walk_metadata, asdf, filename, blocks
  yaml = yaml_serialize(asdf)
  openw, lun, filename, /get_lun
  printf, lun, yaml
  printf, lun, '...'
  free_lun, lun, /force
  if (isa(blocks)) then begin
    asdf_write_append_internal_blocks, filename, blocks
  endif

end